/**
 * 项目：互联网医疗
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.base;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.xwd.hospital.server.annotation.ApiCommon;;
import com.xwd.hospital.server.annotation.AuthUser;
import com.xwd.hospital.server.domain.User;
import com.xwd.hospital.server.rest.res.ApiResponse;
import io.swagger.v3.oas.annotations.Operation;
import org.springframework.web.bind.annotation.*;


import java.util.Arrays;
import java.util.List;

public class BaseController<S extends BaseService<E>, E extends BaseDomain, P extends BaseParam<E>> {

    protected S service;

    /**
     * 通过主键查询
     *
     * @param id 主键
     * @return 单条数据
     */
    @GetMapping("/{id}")
    @Operation(summary = "通过主键查询")
    @ApiCommon
    public ApiResponse<E> selectByPrimaryKey(@PathVariable("id") Long id) {
        return ApiResponse.ok(this.service.getById(id));
    }

    /**
     * 新增
     *
     * @param entity 要新增的对象
     * @return 新增的对象
     */
    @PostMapping("/")
    @Operation(summary = "新增")
    @ApiCommon
    public ApiResponse<E> add(@RequestBody E entity, @AuthUser User user) {
        entity.setDefault().validate(true);
        this.service.save(entity);
        return ApiResponse.ok(this.service.getById(entity.getId()));
    }

    /**
     * 更新
     *
     * @param entity 要更新的对象
     * @return 更新后的对象
     */
    @PutMapping("/")
    @Operation(summary = "更新")
    @ApiCommon
    public ApiResponse<E> update(
            @RequestBody E entity,
            @RequestParam(value = "updateAllFields", defaultValue = "false") boolean updateAllFields,
            @AuthUser User user) {
        if (updateAllFields) {
            this.service.updateAllFieldsById(entity);
        } else {
            this.service.updateById(entity);
        }
        return ApiResponse.ok(this.service.getById(entity.getId()));
    }

    /**
     * 删除
     *
     * @param id 要删除的对象id
     * @return int
     */
    @DeleteMapping("/{id}")
    @Operation(summary = "删除")
    @ApiCommon
    public ApiResponse<Boolean> delete(@PathVariable("id") Long id, @AuthUser User user) {
        return ApiResponse.ok(this.service.removeById(id));
    }

    /**
     * 批量新增
     *
     * @param entityList 要新增的对象
     * @return boolean 成功或失败
     */
    @PostMapping("/batch-save")
    @Operation(summary = "批量新增")
    @ApiCommon
    public ApiResponse<Boolean> batchSave(@RequestBody List<E> entityList, @AuthUser User user) {
        entityList.forEach(entity -> {
            entity.setDefault().validate(true);
        });
        return ApiResponse.ok(this.service.saveBatch(entityList));
    }

    /**
     * 批量更新
     *
     * @param param  更新条件
     * @param entity 要更新的对象
     * @return boolean 成功或失败
     */
    @PostMapping("/batch-update")
    @Operation(summary = "批量更新")
    @ApiCommon
    public ApiResponse<Boolean> batchUpdate(P param, @RequestBody E entity, @AuthUser User user) {
        return ApiResponse.ok(this.service.update(entity, param.toQueryWrapper()));
    }

    /**
     * 批量删除
     *
     * @param idList 要删除的对象id
     * @return boolean 成功或失败
     */
    @PostMapping("/batch-delete")
    @Operation(summary = "批量删除")
    @ApiCommon
    public ApiResponse<Boolean> batchDelete(@RequestBody List<Long> idList, @AuthUser User user) {
        return ApiResponse.ok(this.service.removeByIds(idList));
    }

    /**
     * 分页查询
     *
     * @param param 查询参数
     * @param pageNum pageNum
     * @param pageSize pageSize
     * @return
     */
    @GetMapping("/search")
    @Operation(summary = "分页查询")
    @ApiCommon
    public ApiResponse<IPage<E>> selectByPage(
            P param,
            @RequestParam(value = "orderByClause", defaultValue = "id desc") String orderByClause,
            @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
            @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize
    ) {
        QueryWrapper<E> wrapper = param.toQueryWrapper();
        String[] temp = orderByClause.split("[,;]");
        Arrays.stream(temp).forEach(ob -> {
            String[] obTemp = ob.split("\\s");
            boolean isAsc = obTemp.length == 1 || obTemp[1].equalsIgnoreCase("asc");
            wrapper.orderBy(true, isAsc, obTemp[0]);
        });
        IPage<E> page = new Page<>(pageNum, pageSize);
        return ApiResponse.ok(this.service.page(page, wrapper));
    }

    /**
     * 列表查询
     * @param param
     * @param orderByClause
     * @return
     */
    @GetMapping("/all")
    @Operation(summary = "列表查询")
    @ApiCommon
    public ApiResponse<List<E>> selectByPage(
            P param,
            @RequestParam(value = "orderByClause", defaultValue = "id desc") String orderByClause
    ) {
        QueryWrapper<E> wrapper = param.toQueryWrapper();
        String[] temp = orderByClause.split("[,;]");
        Arrays.stream(temp).forEach(ob -> {
            String[] obTemp = ob.split("\\s");
            boolean isAsc = obTemp.length == 1 || obTemp[1].equalsIgnoreCase("asc");
            wrapper.orderBy(true, isAsc, obTemp[0]);
        });
        return ApiResponse.ok(this.service.list(wrapper));
    }

    /**
     * 单个查询
     *
     * @param param 查询参数
     * @return int
     */
    @GetMapping("/one")
    @Operation(summary = "单个查询")
    @ApiCommon
    public ApiResponse<E> selectOne(P param) {
        QueryWrapper<E> wrapper = param.toQueryWrapper();
        return ApiResponse.ok(this.service.getOne(wrapper, false));
    }

    /**
     * count查询
     *
     * @param param 查询参数
     * @return int
     */
    @GetMapping("/count")
    @Operation(summary = "count查询")
    @ApiCommon
    public ApiResponse<Long> count(P param) {
        QueryWrapper<E> wrapper = param.toQueryWrapper();
        return ApiResponse.ok(this.service.count(wrapper));
    }

}
