/**
 * 项目：互联网医疗
 * 模型分组：系统管理
 * 模型名称：用户表
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.rest;

import java.util.List;
import java.util.Arrays;

import cn.dev33.satoken.secure.SaSecureUtil;
import cn.dev33.satoken.stp.SaLoginModel;
import cn.dev33.satoken.stp.StpUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import com.xwd.hospital.server.annotation.ApiCommon;
import com.xwd.hospital.server.annotation.AuthUser;
import com.xwd.hospital.server.base.BaseController;
import com.xwd.hospital.server.domain.User;
import com.xwd.hospital.server.dto.LoginDto;
import com.xwd.hospital.server.dto.UserDto;
import com.xwd.hospital.server.rest.res.ApiCode;
import com.xwd.hospital.server.domain.User;
import com.xwd.hospital.server.rest.req.UserParam;
import com.xwd.hospital.server.dto.PasswordDto;
import com.xwd.hospital.server.rest.res.ApiResponse;
import com.xwd.hospital.server.service.UserService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springdoc.core.annotations.ParameterObject;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@RestController
@Tag(name = "UserController", description = "用户表管理")
@RequestMapping("/v1/system/user")
public class UserController extends BaseController<UserService, User, UserParam> {

    @Resource
    public void setService(UserService service) {
        this.service = service;
    }

    /**
     * 新增
     *
     * @param entity 要新增的对象
     * @return 新增的对象
     */
    @PostMapping("/")
    @Operation(summary = "新增")
    @Override
    public ApiResponse<User> add(@RequestBody User entity, @AuthUser User user) {
        entity.setEditorId(user.getId());
        entity.setEditorName(user.getUsername());
        if (!StringUtils.isEmpty(entity.getPassword())) {
            entity.setPassword(SaSecureUtil.sha256(entity.getPassword()));
        }
        entity.setDefault().validate(true);
        this.service.save(entity);
        return ApiResponse.ok(this.service.getById(entity.getId()));
    }
    /**
     * 更新
     *
     * @param entity 要更新的对象
     * @return 更新后的对象
     */
    @PutMapping("/")
    @Operation(summary = "更新")
    @Override
    public ApiResponse<User> update(
            @RequestBody User entity,
            @RequestParam(value = "updateAllFields", defaultValue = "false") boolean updateAllFields,
            @AuthUser User user) {
        if (!org.springframework.util.StringUtils.isEmpty(entity.getPassword())) {
            entity.setPassword(SaSecureUtil.sha256(entity.getPassword()));
        }
        if (updateAllFields) {
            this.service.updateAllFieldsById(entity);
        } else {
            this.service.updateById(entity);
        }
        return ApiResponse.ok(this.service.getById(entity.getId()));
    }

    /**
     * 批量新增
     *
     * @param entityList 要新增的对象
     * @return boolean 成功或失败
     */
    @PostMapping("/batch-save")
    @Operation(summary = "批量新增")
    @Override
    public ApiResponse<Boolean> batchSave(@RequestBody List<User> entityList, @AuthUser User user) {
        entityList.forEach(entity -> {
            entity.setEditorId(user.getId());
            entity.setEditorName(user.getUsername());
            entity.setDefault().validate(true);
        });
        return ApiResponse.ok(this.service.saveBatch(entityList));
    }

    /**
     * 分页查询
     *
     * @param param 查询参数
     * @param pageNum pageNum
     * @param pageSize pageSize
     * @return int
     */
    @GetMapping("/search")
    @Operation(summary = "分页查询")
    @Override
    public ApiResponse<IPage<User>> selectByPage(
        UserParam param,
        @RequestParam(value = "orderByClause", defaultValue = "id desc") String orderByClause,
        @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
        @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize
    ) {
        QueryWrapper<User> wrapper = param.toQueryWrapper();
        String[] temp = orderByClause.split("[,;]");
        Arrays.stream(temp).forEach(ob -> {
            String[] obTemp = ob.split("\\s");
            boolean isAsc = obTemp.length == 1 || obTemp[1].equalsIgnoreCase("asc");
            wrapper.orderBy(true, isAsc, obTemp[0]);
        });
        IPage<User> page = new Page<>(pageNum, pageSize);
        return ApiResponse.ok(this.service.page(page, wrapper));
    }
    /**
     * <pre>
     * 用户登陆接口
     * </pre>
     */
    @Operation(summary = "用户登陆接口")
    @PostMapping("/login")
    @ApiCommon
    public ApiResponse<Map> login(@RequestBody User user) {
        Map<String, Object> result = new HashMap<>();
        User loginUser = this.service.login(user.getUsername(), user.getPassword());
        result.put("userInfo", loginUser);
        var loginModel = new SaLoginModel();
        loginModel.setExtra("userId", loginUser.getId());
        StpUtil.login(loginUser.getUsername(), loginModel);
        result.put("token", StpUtil.getTokenValue());
        return ApiResponse.ok(result);
    }

    /**
     * <pre>
     * 用户登陆接口
     * </pre>
     */
    @Operation(summary = "小程序用户登陆接口")
    @PostMapping("/miniAppLogin")
    @ApiCommon
    public ApiResponse<UserDto> miniAppLogin(@RequestBody LoginDto loginDto) {

        UserDto loginUser = this.service.miniAppLogin(loginDto);

        return ApiResponse.ok(loginUser);
    }

     /**
     * <pre>
     * 用户登出接口
     * </pre>
     */
    @Operation(summary = "用户登出接口")
    @PostMapping("/logout")
    @ApiCommon
    public ApiResponse<Object> logout() {
        StpUtil.logout();
        return ApiResponse.ok(null);
    }

    /**
     * <pre>
     * 用户注册接口
     * </pre>
     */
    @Operation(summary = "用户注册接口")
    @PostMapping("/register")
    @ApiCommon
    public ApiResponse<User> register(@RequestBody User user) {
        return ApiResponse.ok(this.service.register(user, "ROLE_USER"));
    }

    /**
     * <pre>
     * 查询当前登录的用户信息
     * </pre>
     */
    @Operation(summary = "查询当前登录的用户信息")
    @GetMapping("/info")
    @ApiCommon
    public ApiResponse<User> currentUserInfo(@AuthUser(queryDb = true) User user) {
        return ApiResponse.ok(user);
    }

    @PostMapping("/change-password")
    @Operation(summary = "修改密码")
    @ApiCommon
    public ApiResponse<User> updatePassword(@RequestBody PasswordDto passwordDto) {
        User user = this.service.getById(passwordDto.getUserId());
        if (null != user) {
            if (user.getPassword().equals(SaSecureUtil.sha256(passwordDto.getOldPassword()))){
                user.setPassword(SaSecureUtil.sha256(passwordDto.getNewPassword()));
                this.service.updateById(user);
            } else {
                return ApiResponse.fail(-1, "原密码不正确");
            }
        }
        return ApiResponse.ok(user);
    }

}
