/**
 * 项目：互联网医疗
 * 模型分组：服务管理
 * 模型名称：订单表
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.service.impl;

import java.io.Serializable;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

import cn.dev33.satoken.stp.StpUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.xwd.hospital.server.domain.*;
import com.xwd.hospital.server.dto.CreateOrderDto;
import com.xwd.hospital.server.dto.OrderDto;
import com.xwd.hospital.server.enums.OrderStateEnum;
import com.xwd.hospital.server.enums.OrderTypeEnum;
import com.xwd.hospital.server.enums.YesNoEnum;
import com.xwd.hospital.server.service.*;
import jakarta.annotation.Resource;
import com.baomidou.mybatisplus.core.enums.SqlMethod;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.baomidou.mybatisplus.extension.toolkit.SqlHelper;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import com.xwd.hospital.server.repository.OrderMapper;
import com.xwd.hospital.server.repository.base.OrderBaseMapper;
import com.xwd.hospital.server.rest.req.OrderParam;

@Service
public class OrderServiceImpl extends ServiceImpl<OrderMapper, Order> implements OrderService {
    @Resource
    private UserInfoService userInfoService;
    @Resource
    private DoctorServiceInfoService doctorServiceInfoService;
    @Resource
    private UserCouponInfoService userCouponInfoService;
    @Resource
    private DoctorInfoService doctorInfoService;
    @Resource
    private PatientInfoService patientInfoService;

    @Override
    public int updateAllFieldsById(Order entity) {
        return this.getBaseMapper().updateAllFieldsById(entity);
    }

    /**
     * 批量插入
     *
     * @param entityList ignore
     * @param batchSize  ignore
     * @return ignore
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveBatch(Collection<Order> entityList, int batchSize) {
        String sqlStatement = SqlHelper.getSqlStatement(OrderBaseMapper.class, SqlMethod.INSERT_ONE);
        return executeBatch(entityList, batchSize, (sqlSession, entity) -> sqlSession.insert(sqlStatement, entity));
    }

    @Override
    public List<OrderDto> queryOrderListForMiniApp(OrderStateEnum param) {
        long userId = StpUtil.getLoginIdAsLong();
        UserInfo userInfo = userInfoService.getOne(Wrappers.<UserInfo>query().eq("user_id", userId));

        return this.getBaseMapper().queryOrderListForMiniApp(userInfo.getId(),param.toString());
    }


    @Override
    public Order createOrder(CreateOrderDto createOrderDto) {
        Order order = new Order();
        //写入订单相关记录记录
        order.setUserInfoId(createOrderDto.getUserInfoId());
        order.setDoctorId(createOrderDto.getDoctorId());
        order.setPatientId(createOrderDto.getPatientId());
        //生成订单号
        //todo 订单号生成规则待定
        order.setOrderNo(geneartOrderNo());
        order.setOrderState(OrderStateEnum.TO_PAY);
        order.setOrderType(createOrderDto.getOrderType());
        order.setAppointmentDate(createOrderDto.getAppointmentDate());

        //查询相关服务价格
        OrderTypeEnum orderType = createOrderDto.getOrderType();
        Long doctorId = createOrderDto.getDoctorId();
        DoctorInfo doctorInfo = doctorInfoService.getById(doctorId);

        DoctorServiceInfo doctorServiceInfo = doctorServiceInfoService.getOne(
                Wrappers.<DoctorServiceInfo>query().eq("doctor_id", doctorId)
                        .eq("service_name", orderType.toString())
                        .eq("service_state", YesNoEnum.YES.toString()));
        if(null == doctorServiceInfo){

        }
        order.setOrderPrice(doctorServiceInfo.getServicePrice());

        //查询优惠金额
        order.setOrderDiscount(BigDecimal.ZERO);
        if(null != createOrderDto.getUserCouponId()){
            UserCouponInfo userCouponInfo = userCouponInfoService.getById(createOrderDto.getUserCouponId());
            //校验优惠券有效性
            if(checkCoupon(userCouponInfo)){
                order.setOrderDiscount(userCouponInfo.getCouponDiscount());
                order.setUserCouponId(createOrderDto.getUserCouponId());

                //优惠券变为已使用
                userCouponInfo.setCouponUseState(YesNoEnum.NO);
                userCouponInfoService.save(userCouponInfo);
            }
        }

        //计算实际价格
        BigDecimal caulatePrice = order.getOrderPrice().subtract(order.getOrderDiscount());
        order.setOrderActualPay(caulatePrice.compareTo(BigDecimal.ZERO) > 0?order.getOrderPrice().subtract(order.getOrderDiscount()) : BigDecimal.ZERO );

        //写入服务相关信息
        order.setServiceStartTime(doctorServiceInfo.getServiceStartTime());
        order.setServiceEndTime(doctorServiceInfo.getServiceEndTime());
        order.setDoctorName(doctorInfo.getDoctorName());
        order.setHospitalName(doctorInfo.getMedicalQualification().getHospital().getHospitalName());
        order.setDepartmentName(doctorInfo.getMedicalQualification().getDepartment().getDepartmentName());

        //写入就诊人相关信息
        PatientInfo patientInfo = patientInfoService.getById(createOrderDto.getPatientId());
        order.setPatientName(patientInfo.getPatientName());
        order.setPatientSex(patientInfo.getSex());
        order.setPatientIdNo(patientInfo.getIdNo());
        order.setPatientPhoneNumber(patientInfo.getPhoneNumber());
        order.setPatientBornDate(patientInfo.getBornDate());
        order.setPatientSelfIntroduce(createOrderDto.getPatientSelfIntroduce());
        this.save(order);

        return order;
    }

    /**
     * 生成订单
     * @return
     */
    public static String geneartOrderNo() {
        //生成方式：当前时间+随机6位数字
        // 获取当前日期时间
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
        String dateTime = dateFormat.format(new Date());

        // 生成6位随机数
        Random random = new Random();
        int randomValue = random.nextInt(900000) + 100000; // 生成100000到999999之间的随机数

        // 将时间戳和随机数拼接成订单号
        String serialNumber = String.valueOf(dateTime) + String.valueOf(randomValue);

        return serialNumber;
    }

    /**
     * 校验优惠券有效性
     * @return
     */
    public static boolean checkCoupon(UserCouponInfo userCouponInfo) {
        //校验优惠券可用状态
        if(userCouponInfo.getCouponUseState() == YesNoEnum.NO){
            return false;
        }
        //校验优惠券有效日期
        Date today = new Date();

        if(today.after(userCouponInfo.getCouponStartDate()) && today.before(userCouponInfo.getCouponEndDate())){
            return true;
        }else {
            return false;
        }
    }

    @Override
    public void cancelOrder(String orderNo) {
        //修改订单状态
        Order order = this.getOne(Wrappers.<Order>query().eq("order_no", orderNo));
        if(order.getOrderState() == OrderStateEnum.TO_PAY){
            order.setOrderState(OrderStateEnum.CANCELED);
            this.save(order);
        }

    }
}
