package com.xwd.hospital.server.service.impl;

import cn.dev33.satoken.stp.StpUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.wechat.pay.java.core.Config;
import com.wechat.pay.java.core.RSAAutoCertificateConfig;
import com.wechat.pay.java.core.exception.HttpException;
import com.wechat.pay.java.core.exception.MalformedMessageException;
import com.wechat.pay.java.core.exception.ServiceException;
import com.wechat.pay.java.core.exception.ValidationException;
import com.wechat.pay.java.core.notification.NotificationConfig;
import com.wechat.pay.java.core.notification.NotificationParser;
import com.wechat.pay.java.core.notification.RequestParam;
import com.wechat.pay.java.service.payments.jsapi.JsapiService;
import com.wechat.pay.java.service.payments.jsapi.model.*;
import com.wechat.pay.java.service.payments.jsapi.model.Amount;
import com.wechat.pay.java.service.payments.model.Transaction;
import com.wechat.pay.java.service.payments.nativepay.NativePayService;
import com.wechat.pay.java.service.refund.RefundService;
import com.wechat.pay.java.service.refund.model.*;
import com.xwd.hospital.server.domain.Order;
import com.xwd.hospital.server.domain.OrderPay;
import com.xwd.hospital.server.domain.OrderRefund;
import com.xwd.hospital.server.enums.OrderStateEnum;
import com.xwd.hospital.server.enums.PayPathEnum;
import com.xwd.hospital.server.enums.PayStateEnum;
import com.xwd.hospital.server.enums.RefundStateEnum;
import com.xwd.hospital.server.service.OrderPayService;
import com.xwd.hospital.server.service.OrderRefundService;
import com.xwd.hospital.server.service.OrderService;
import com.xwd.hospital.server.service.PaymentService;
import jakarta.annotation.Resource;
import jakarta.servlet.ServletInputStream;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.sql.Wrapper;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Random;
import java.util.TimeZone;

@Service
@Slf4j
public class PaymentServiceImpl implements PaymentService {

    public static JsapiService payService;
    public static RefundService refundService;

    @Resource
    private OrderService orderService;
    @Resource
    private OrderPayService orderPayService;
    @Resource
    private OrderRefundService orderRefundService;


    /** 商户号 */
    @Value("$(pay.wx.merchantId)")
    public static String merchantId = "190000****";
    /** 商户API私钥路径 */
    @Value("$(pay.wx.keyPath)")
    public static String privateKeyPath = "/path/apiclient_key.pem";
    /** 商户证书序列号 */
    @Value("$(pay.wx.merchantSerialNumber)")
    public static String merchantSerialNumber = "5157F09EFDC096DE15EBE81A47057A72********";
    /** 商户APIV3密钥 */
    @Value("$(pay.wx.apiV3Key)")
    public static String apiV3Key = "...";
    /** appId */
    @Value("$(pay.wx.appId)")
    public static String appId = "...";
    /** 支付回调地址 */
    @Value("$(pay.wx.payNotifyUrl)")
    public static String payNotifyUrl = "...";

    @Value("$(pay.wx.refundNotifyUrl)")
    public static String refundNotifyUrl = "...";

    public static int PAY_DELAY_TIME = 30;

    @Override
    public String startPay(String orderNo, PayPathEnum payPath) {
        if(payPath.equals(PayPathEnum.WX_PAY)){
            //微信支付
            // 使用自动更新平台证书的RSA配置
            // 初始化商户配置
            Config config =
                    new RSAAutoCertificateConfig.Builder()
                            .merchantId(merchantId)
                            // 使用 com.wechat.pay.java.core.util 中的函数从本地文件中加载商户私钥，商户私钥会用来生成请求的签名
                            .privateKeyFromPath(privateKeyPath)
                            .merchantSerialNumber(merchantSerialNumber)
                            .apiV3Key(apiV3Key)
                            .build();

            // 初始化服务
            payService = new JsapiService.Builder().config(config).build();

            Order order = orderService.getOne(Wrappers.<Order>query().eq("order_no", orderNo));
            if(order.getOrderState() != OrderStateEnum.TO_PAY){
                return null;
            }
            //判断是否调用过支付接口
            OrderPay orderPay = orderPayService.getOne(Wrappers.<OrderPay>query().eq("order_no", orderNo).eq("pay_state",PayStateEnum.TO_PAY.toString()));
            if (null != orderPay){
                closeOrder(orderPay);
                orderPay.setPayNo(geneartPayNo());
            }else {
                //向order_pay插入新的pay记录
                orderPay = new OrderPay();
                orderPay.setOrderId(order.getId());
                orderPay.setPayNo(geneartPayNo());
                orderPay.setOrderNo(orderNo);
                orderPay.setPayState(PayStateEnum.TO_PAY);
                orderPay.setPayPath(PayPathEnum.WX_PAY);
                orderPay.setPayMoney(order.getOrderActualPay());
                orderPay.setEditorId(StpUtil.getLoginIdAsLong());
            }

                orderPay.setPayTime(new Date());
                orderPayService.save(orderPay);


            //  调用接口
            try {
                PrepayResponse response = prepay(order, orderPay);
                return response.getPrepayId();
            } catch (HttpException e) { // 发送HTTP请求失败
                // 调用e.getHttpRequest()获取请求打印日志或上报监控，更多方法见HttpException定义
            } catch (ServiceException e) { // 服务返回状态小于200或大于等于300，例如500
                // 调用e.getResponseBody()获取返回体打印日志或上报监控，更多方法见ServiceException定义
            } catch (MalformedMessageException e) { // 服务返回成功，返回体类型不合法，或者解析返回体失败
                // 调用e.getMessage()获取信息打印日志或上报监控，更多方法见MalformedMessageException定义
            }
        }else if(payPath.equals(PayPathEnum.ALI_PAY)){

        }
        return null;
    }

    @Override
    public void payNotify(HttpServletRequest request, HttpServletResponse response) {
        //获取报文
        String requestBody = getRequestBody(request);
        //随机串
        String wechatpayNonce = request.getHeader("Wechatpay-Nonce");

        //微信传递过来的签名
        String wechatSignature = request.getHeader("Wechatpay-Signature");

        //证书序列号（微信平台）
        String wechatPaySerial = request.getHeader("Wechatpay-Serial");

        //时间戳
        String wechatTimestamp = request.getHeader("Wechatpay-Timestamp");

        // 构造 RequestParam
        RequestParam requestParam = new RequestParam.Builder()
                .serialNumber(wechatPaySerial)
                .nonce(wechatpayNonce)
                .signature(wechatSignature)
                .timestamp(wechatTimestamp)
                .body(requestBody)
                .build();

        // 如果已经初始化了 RSAAutoCertificateConfig，可直接使用
        // 没有的话，则构造一个
        NotificationConfig config = new RSAAutoCertificateConfig.Builder()
                .merchantId(merchantId)
                .privateKeyFromPath(privateKeyPath)
                .merchantSerialNumber(merchantSerialNumber)
                .apiV3Key(apiV3Key)
                .build();

        // 初始化 NotificationParser
        NotificationParser parser = new NotificationParser(config);

        try {
            // 以支付通知回调为例，验签、解密并转换成 Transaction
            Transaction transaction = parser.parse(requestParam, Transaction.class);
            //记录日志信息
            Transaction.TradeStateEnum state = transaction.getTradeState();
            if (!StringUtils.equals("SUCCESS", state.toString())) {
                log.error("微信回调失败,JsapiPayController.payNotify.transaction：{}",transaction.toString());
                //通知微信回调失败
                response.getWriter().write("<xml><return_code><![CDATA[FAIL]]></return_code></xml>");
            }else {
                //todo 处理订单状态等逻辑
                String outTradeNo = transaction.getOutTradeNo();
                OrderPay orderPay = orderPayService.getOne(Wrappers.<OrderPay>query().eq("pay_no", outTradeNo));
                orderPay.setPayState(PayStateEnum.PAYED);
                orderPay.setPayTime(new Date());
                orderPayService.save(orderPay);

                Order order = orderService.getById(orderPay.getOrderId());
                order.setOrderState(OrderStateEnum.TO_CONFIRM);
                order.setPayTime(new Date());
                orderService.save(order);


                response.getWriter().write("<xml><return_code><![CDATA[SUCCESS]]></return_code></xml>");
            }

        } catch (Exception e) {
            // 验证失败
            log.error("wxPay notify failed", e);
        }
    }

    @Override
    public void refundNotify(HttpServletRequest request, HttpServletResponse response) {
        //获取报文
        String requestBody = getRequestBody(request);
        //随机串
        String wechatpayNonce = request.getHeader("Wechatpay-Nonce");

        //微信传递过来的签名
        String wechatSignature = request.getHeader("Wechatpay-Signature");

        //证书序列号（微信平台）
        String wechatPaySerial = request.getHeader("Wechatpay-Serial");

        //时间戳
        String wechatTimestamp = request.getHeader("Wechatpay-Timestamp");

        // 构造 RequestParam
        RequestParam requestParam = new RequestParam.Builder()
                .serialNumber(wechatPaySerial)
                .nonce(wechatpayNonce)
                .signature(wechatSignature)
                .timestamp(wechatTimestamp)
                .body(requestBody)
                .build();

        // 如果已经初始化了 RSAAutoCertificateConfig，可直接使用
        // 没有的话，则构造一个
        NotificationConfig config = new RSAAutoCertificateConfig.Builder()
                .merchantId(merchantId)
                .privateKeyFromPath(privateKeyPath)
                .merchantSerialNumber(merchantSerialNumber)
                .apiV3Key(apiV3Key)
                .build();

        // 初始化 NotificationParser
        NotificationParser parser = new NotificationParser(config);

        try {
            // 以支付通知回调为例，验签、解密并转换成 RefundNotification
            RefundNotification refundNotification = parser.parse(requestParam, RefundNotification.class);

            //记录日志信息
            Status status = refundNotification.getRefundStatus();
            if (!StringUtils.equals("SUCCESS", status.toString())) {
                log.error("微信回调失败,JsapiPayController.refundNotification：{}",refundNotification.toString());
                //通知微信回调失败
                response.getWriter().write("<xml><return_code><![CDATA[FAIL]]></return_code></xml>");
            }else {
                //todo 处理退款逻辑
                String refundNo = refundNotification.getOutRefundNo();
                OrderRefund orderRefund = orderRefundService.getOne(Wrappers.<OrderRefund>query().eq("refund_no", refundNo));
                orderRefund.setRefundState(RefundStateEnum.REFUNDED);
                orderRefund.setRefundTime(new Date());
                orderRefundService.save(orderRefund);

                response.getWriter().write("<xml><return_code><![CDATA[SUCCESS]]></return_code></xml>");
            }

        } catch (Exception e) {
            // 验证失败
            log.error("wxPay notify failed", e);
        }
    }

    @Override
    public void refundPay(String orderNo) {
        // 初始化商户配置
        Config config =
                new RSAAutoCertificateConfig.Builder()
                        .merchantId(merchantId)
                        // 使用 com.wechat.pay.java.core.util 中的函数从本地文件中加载商户私钥，商户私钥会用来生成请求的签名
                        .privateKeyFromPath(privateKeyPath)
                        .merchantSerialNumber(merchantSerialNumber)
                        .apiV3Key(apiV3Key)
                        .build();

        // 初始化服务
        refundService = new RefundService.Builder().config(config).build();

        OrderPay orderPay = orderPayService.getOne(Wrappers.<OrderPay>query().eq("order_no", orderNo));
        OrderRefund orderRefund = new OrderRefund();
        orderRefund.setOrderId(orderPay.getOrderId());
        orderRefund.setOrderNo(orderNo);
        orderRefund.setOrderPayId(orderPay.getId());
        orderRefund.setRefundNo(geneartPayNo());
        orderRefund.setRefundState(RefundStateEnum.REFUNDING);
        orderRefund.setRefundMoney(orderPay.getPayMoney());
        orderRefund.setRefundTime(new Date());
        orderRefundService.save(orderRefund);
        // ... 调用接口
        try {
            Refund response = refundCreate(orderRefund,orderPay.getPayNo());
        } catch (HttpException e) { // 发送HTTP请求失败
            // 调用e.getHttpRequest()获取请求打印日志或上报监控，更多方法见HttpException定义
        } catch (ServiceException e) { // 服务返回状态小于200或大于等于300，例如500
            // 调用e.getResponseBody()获取返回体打印日志或上报监控，更多方法见ServiceException定义
        } catch (MalformedMessageException e) { // 服务返回成功，返回体类型不合法，或者解析返回体失败
            // 调用e.getMessage()获取信息打印日志或上报监控，更多方法见MalformedMessageException定义
        }

    }

    /**
     * 生成支付/退款流水号
     * @return
     */
    public static String geneartPayNo() {
        //生成方式：当前时间+随机6位数字
        // 获取当前日期时间
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
        String dateTime = dateFormat.format(new Date());

        // 生成6位随机数
        Random random = new Random();
        int randomValue = random.nextInt(900000) + 100000; // 生成100000到999999之间的随机数

        // 将时间戳和随机数拼接成流水号
        String serialNumber = String.valueOf(dateTime) + String.valueOf(randomValue);

        return serialNumber;
    }


    /** 关闭订单 */
    public static void closeOrder(OrderPay orderPay) {

        CloseOrderRequest request = new CloseOrderRequest();
        // 调用request.setXxx(val)设置所需参数，具体参数可见Request定义
        request.setOutTradeNo(orderPay.getPayNo());
        request.setMchid(merchantId);
        // 调用接口
        payService.closeOrder(request);
    }
    /** JSAPI支付下单 */
    public static PrepayResponse prepay(Order order,OrderPay orderPay) {

        PrepayRequest request = new PrepayRequest();
        // 调用request.setXxx(val)设置所需参数，具体参数可见Request定义
        request.setAppid(appId);
        request.setMchid(merchantId);
        request.setDescription(order.getOrderType().getName());
        request.setOutTradeNo(orderPay.getPayNo());

        //订单支付时间为x分钟
        // 创建一个 Calendar 对象，并设置其时间为当前时间
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(orderPay.getPayTime());
        // 将时间增加x分钟
        calendar.add(Calendar.MINUTE, PAY_DELAY_TIME);
        Date newDate = calendar.getTime();
        // 格式化 Date 对象为 RFC3339 格式字符串
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
        sdf.setTimeZone(TimeZone.getTimeZone("Asia/Shanghai"));
        String expireTime = sdf.format(newDate);
        request.setTimeExpire(expireTime);

        request.setNotifyUrl(payNotifyUrl);
        //支付金额
        Amount amount = new Amount();
        amount.setCurrency("CNY");
        amount.setTotal(order.getOrderActualPay().multiply(new BigDecimal(100)).intValue());
        request.setAmount(amount);
        //支付用户
        Payer payer = new Payer();
        payer.setOpenid(order.getUserInfo().getUser().getOpenId());
        request.setPayer(payer);
        // 调用接口
        return payService.prepay(request);
    }
    /** 微信支付订单号查询订单 */
    public static Transaction queryOrderById() {

        QueryOrderByIdRequest request = new QueryOrderByIdRequest();
        // 调用request.setXxx(val)设置所需参数，具体参数可见Request定义
        // 调用接口
        return payService.queryOrderById(request);
    }
    /** 商户订单号查询订单 */
    public static Transaction queryOrderByOutTradeNo() {

        QueryOrderByOutTradeNoRequest request = new QueryOrderByOutTradeNoRequest();
        // 调用request.setXxx(val)设置所需参数，具体参数可见Request定义
        // 调用接口
        return payService.queryOrderByOutTradeNo(request);
    }

    /** 退款申请 */
    public static Refund refundCreate(OrderRefund orderRefund,String payNo) {
        CreateRequest request = new CreateRequest();
        // 调用request.setXxx(val)设置所需参数，具体参数可见Request定义
        request.setOutTradeNo(payNo);
        request.setOutRefundNo(orderRefund.getRefundNo());
        request.setNotifyUrl(refundNotifyUrl);

        //全额退款
        AmountReq amountReq = new AmountReq();
        amountReq.setRefund(orderRefund.getRefundMoney().multiply(new BigDecimal(100)).longValue());
        amountReq.setCurrency("CNY");
        amountReq.setTotal(orderRefund.getRefundMoney().multiply(new BigDecimal(100)).longValue());
        request.setAmount(amountReq);
        // 调用接口
        return refundService.create(request);
    }
    /** 查询单笔退款（通过商户退款单号） */
    public static Refund queryByOutRefundNo() {

        QueryByOutRefundNoRequest request = new QueryByOutRefundNoRequest();
        // 调用request.setXxx(val)设置所需参数，具体参数可见Request定义
        // 调用接口
        return refundService.queryByOutRefundNo(request);
    }


    /**
     * 读取请求数据流
     * @param request
     * @return
     */
    private String getRequestBody(HttpServletRequest request) {
        StringBuffer sb = new StringBuffer();
        try (ServletInputStream inputStream = request.getInputStream();
             BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
        ) {
            String line;

            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }
        } catch (IOException e) {
            log.error("读取数据流异常:{}", e);
        }
        return sb.toString();
    }

}
