/**
 * 项目：互联网医疗
 * 模型分组：系统管理
 * 模型名称：用户表
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.service.impl;

import java.io.Serializable;
import java.util.List;
import java.util.ArrayList;
import java.util.Collection;

import cn.binarywang.wx.miniapp.api.WxMaService;
import cn.binarywang.wx.miniapp.bean.WxMaJscode2SessionResult;
import cn.binarywang.wx.miniapp.bean.WxMaUserInfo;
import cn.binarywang.wx.miniapp.util.WxMaConfigHolder;
import com.xwd.hospital.server.constants.Constant;
import com.xwd.hospital.server.domain.*;
import com.xwd.hospital.server.dto.LoginDto;
import com.xwd.hospital.server.dto.RegisterDto;
import com.xwd.hospital.server.dto.UpdateUserDto;
import com.xwd.hospital.server.dto.UserDto;
import com.xwd.hospital.server.enums.RegisterSourceEnum;
import com.xwd.hospital.server.enums.UserTypeEnum;
import com.xwd.hospital.server.service.*;
import jakarta.annotation.Resource;
import com.baomidou.mybatisplus.core.enums.SqlMethod;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.baomidou.mybatisplus.extension.toolkit.SqlHelper;
import me.chanjar.weixin.common.error.WxErrorException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import com.xwd.hospital.server.repository.UserMapper;
import com.xwd.hospital.server.repository.base.UserBaseMapper;
import com.xwd.hospital.server.rest.req.UserParam;
import com.xwd.hospital.server.rest.res.ApiCode;
import cn.dev33.satoken.secure.SaSecureUtil;
import com.xwd.hospital.server.repository.RoleMapper;
import com.xwd.hospital.server.enums.YesNoEnum;
import com.xwd.hospital.server.repository.UserRoleMapper;

@Service
public class UserServiceImpl extends ServiceImpl<UserMapper, User> implements UserService {
    @Resource
    private UserRoleMapper userRoleMapper;
    @Resource
    private RoleMapper roleMapper;
    @Resource
    private WxMaService wxMaService;
    @Resource
    private RedisService redisService;
    @Resource
    private UserInfoService userInfoService;
    @Resource
    private DoctorInfoService doctorInfoService;

    @Value("$(wx.miniapp.configs[0].appid)")
    private String patientAppId;

    @Value("$(wx.miniapp.configs[1].appid)")
    private String doctorAppId;

    @Override
    public int updateAllFieldsById(User entity) {
        return this.getBaseMapper().updateAllFieldsById(entity);
    }

    /**
     * 批量插入
     *
     * @param entityList ignore
     * @param batchSize  ignore
     * @return ignore
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveBatch(Collection<User> entityList, int batchSize) {
        String sqlStatement = SqlHelper.getSqlStatement(UserBaseMapper.class, SqlMethod.INSERT_ONE);
        return executeBatch(entityList, batchSize, (sqlSession, entity) -> sqlSession.insert(sqlStatement, entity));
    }

    @Override
    public User login(String username, String password) {
        User user = this.selectByUsername(username);
        if (user == null) {
            throw new ApiCode.ApiException(-1, "用户名不存在！");
        }
        if (!SaSecureUtil.sha256(password).equals(user.getPassword())) {
            throw new ApiCode.ApiException(-3, "用户名和密码不匹配！");
        }
        return user;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public User register(User user, String roleName) {
        user.setDefault().validate(true);
        if (user.getPassword() == null) {
            throw new ApiCode.ApiException(-1, "用户密码不能为空！");
        }
        user.setPassword(SaSecureUtil.sha256(user.getPassword()));
        // 判断用户名是否存在
        User existUser = this.selectByUsername(user.getUsername());
        if (existUser != null) {
            throw new ApiCode.ApiException(-2, "用户已注册，请登录！");
        }
        this.save(user);
        // 分配默认角色给用户
        Role defaultRole = this.roleMapper.selectOne(Wrappers.<Role>lambdaQuery().eq(Role::getName, roleName));
        if (defaultRole != null) {
            this.userRoleMapper.insert(UserRole.newBuilder().userId(user.getId()).roleId(defaultRole.getId()).build());
        }
        return this.getById(user.getId());
    }

    @Override
    public User selectByUsername(String username) {
        return this.getOne(Wrappers.<User>query().eq("username", username));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean save(User entity) {
        boolean result = super.save(entity);
        if (entity.getRoleList() != null) {
            this.userRoleMapper.delete(Wrappers.<UserRole>query().eq("user_id", entity.getId()));
            List<UserRole> userRoleList = new ArrayList<>();
            for (Role role: entity.getRoleList()) {
                UserRole userRole = new UserRole();
                userRole.setUserId(entity.getId());
                userRole.setRoleId(role.getId());
                userRoleList.add(userRole);
            }
            if (userRoleList.size() > 0) {
                this.userRoleMapper.batchInsert(userRoleList);
            }
        }
        return result;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateById(User entity) {
        boolean result = super.updateById(entity);
        if (entity.getRoleList() != null) {
            this.userRoleMapper.delete(Wrappers.<UserRole>query().eq("user_id", entity.getId()));
            List<UserRole> userRoleList = new ArrayList<>();
            for (Role role: entity.getRoleList()) {
                UserRole userRole = new UserRole();
                userRole.setUserId(entity.getId());
                userRole.setRoleId(role.getId());
                userRoleList.add(userRole);
            }
            if (userRoleList.size() > 0) {
                this.userRoleMapper.batchInsert(userRoleList);
            }
        }
        return result;
    }

    @Override
    public UserDto miniAppLogin(LoginDto loginDto){
        UserDto userDto = new UserDto();
        //短信验证码登录
        if (StringUtils.isBlank(loginDto.getCode())) {
            String authCode = redisService.get(Constant.SMS_PREFIX + loginDto.getPhoneNumber());
            if(StringUtils.isBlank(authCode) ||
                    StringUtils.isBlank(loginDto.getSmsAuthCode()) ||
                    !authCode.equals(loginDto.getSmsAuthCode())){
                //校验不通过,直接返回空userDto
                userDto.setAuthFlag(false);
                return userDto;
            }else {
                userDto.setAuthFlag(true);
                User user = this.getOne(Wrappers.<User>query().eq("phone", loginDto.getPhoneNumber()));
                if(null == user){
                    //此手机号尚未在系统中注册，直接注册为新用户
                    user = new User();
                    user.setUsername(loginDto.getPhoneNumber());
                    user.setUserType(loginDto.getUserType());
                    user.setRegisterSource(RegisterSourceEnum.WX);
                    user.setPhone(loginDto.getPhoneNumber());
                    this.save(user);
                }
                userDto.setUser(user);
                return userDto;
            }
        }else {//微信直接授权登录
            //根据小程序登录源，切换相应配置
            if(loginDto.getUserType() == UserTypeEnum.PATIENT){
                wxMaService.switchover(patientAppId);
            }else if(loginDto.getUserType() == UserTypeEnum.DOCTOR){
                wxMaService.switchover(doctorAppId);
            }
            try {
                WxMaJscode2SessionResult session = wxMaService.getUserService().getSessionInfo(loginDto.getCode());
                log.debug(session.getSessionKey());
                log.debug(session.getOpenid());
                //通过openId与sys_user表关联 判断是否需要进行注册流程
                String openid = session.getOpenid();
                User user = this.getOne(Wrappers.<User>query().eq("open_id", openid));
                if(null == user){//open_id未查询到相关用户
                    userDto.setRegisterFlag(true);
                    return userDto;
                }else{
                    userDto.setUser(user);
                    if(null != user.getObjId()){//关联信息表
                        if(user.getUserType() == UserTypeEnum.PATIENT){
                            userDto.setUserInfo(userInfoService.getById(user.getObjId()));
                        }else if(loginDto.getUserType() == UserTypeEnum.DOCTOR){
                            userDto.setDoctorInfo(doctorInfoService.getById(user.getObjId()));
                        }
                    }
                    return userDto;
                }
            } catch (WxErrorException e) {
                log.error("微信登录异常");
            } finally {
                WxMaConfigHolder.remove();//清理ThreadLocal
            }
        }
        return userDto;
    };

    @Override
    public UserDto miniAppRegister(RegisterDto registserDto) {
        UserDto userDto = new UserDto();
        //根据小程序登录源，切换相应配置
        if(registserDto.getUserType() == UserTypeEnum.PATIENT){
            wxMaService.switchover(patientAppId);
        }else if(registserDto.getUserType() == UserTypeEnum.DOCTOR){
            wxMaService.switchover(doctorAppId);
        }
        try {
            WxMaJscode2SessionResult session = wxMaService.getUserService().getSessionInfo(registserDto.getCode());
            log.debug(session.getSessionKey());
            log.debug(session.getOpenid());
            //通过openId与sys_user表关联 判断是否需要进行注册流程
            String openid = session.getOpenid();
            User user = this.getOne(Wrappers.<User>query().eq("phone", registserDto.getPhoneNumber()));
            if(null != user ){
                user.setOpenId(openid);
            }else {
                //尚未在系统中注册，直接注册为新用户
                user = new User();
                user.setUsername(registserDto.getPhoneNumber());
                user.setUserType(registserDto.getUserType());
                user.setRegisterSource(RegisterSourceEnum.WX);
                user.setPhone(registserDto.getPhoneNumber());
                user.setOpenId(openid);
            }
            this.save(user);
            return userDto;
        } catch (WxErrorException e) {
            log.error("微信登录异常");
        } finally {
            WxMaConfigHolder.remove();//清理ThreadLocal
        }
        return userDto;
    }

    @Override
    public UserDto updateMiniUserInfo(UpdateUserDto updateUserDto) {
        UserDto userDto = new UserDto();

        //根据小程序登录源，切换相应配置
        if(updateUserDto.getUserType() == UserTypeEnum.PATIENT){
            wxMaService.switchover(patientAppId);
        }else if(updateUserDto.getUserType() == UserTypeEnum.DOCTOR){
            wxMaService.switchover(doctorAppId);
        }
        try {
            WxMaJscode2SessionResult session = wxMaService.getUserService().getSessionInfo(updateUserDto.getCode());
            log.debug(session.getSessionKey());
            log.debug(session.getOpenid());
            User user = this.getOne(Wrappers.<User>query().eq("open_id", session.getOpenid()));
            WxMaUserInfo userInfo = wxMaService.getUserService().getUserInfo(session.getSessionKey(), updateUserDto.getEncryptedData(), updateUserDto.getIvStr());
            user.setUsername(userInfo.getNickName());
            user.setAvatar(userInfo.getAvatarUrl());
            this.save(user);
            userDto.setUser(user);
            return userDto;
        } catch (WxErrorException e) {
            log.error("微信登录异常");
        } finally {
            WxMaConfigHolder.remove();//清理ThreadLocal
        }
        return userDto;
    }
}
