/**
 * 项目：互联网医疗
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.ws;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.xwd.hospital.server.util.SpringUtils;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.socket.*;

import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

@Slf4j
public class WebSocketService implements WebSocketHandler {
    /**
     * 当前会话集合
     */
    private static ConcurrentMap<String, WebSocketSession> sessions = new ConcurrentHashMap<>();

    /**
     * 链接成功调用的方法
     */
    @Override
    public void afterConnectionEstablished(WebSocketSession session) {
        String username = (String) session.getAttributes().get("username");
        sessions.put(username, session);
        log.info("【websocket】用户连接:" + username);
    }

    private void closeSession(WebSocketSession session) {
        if (session != null && session.isOpen()) {
            try {
                session.close();
            } catch (IOException e) {
                log.error("【websocket】关闭会话异常", e);
            }
        }
    }

    @SneakyThrows
    private void sendMessage(WebSocketSession session, Object obj) {
        if (obj != null) {
            ObjectMapper objectMapper = SpringUtils.getBean(ObjectMapper.class);
            String message = objectMapper.writeValueAsString(obj);
            if (session != null && session.isOpen()) {
                session.sendMessage(new TextMessage(message));
                log.info("【websocket】发送消息：{}", message);
            } else {
                log.warn("【websocket】会话已关闭，不发送消息：{}", message);
            }
        }
    }

    /**
     * 收到客户端消息
     * @param message
     */
    @Override
    public void handleMessage(WebSocketSession session, WebSocketMessage<?> message) {
        // TODO
        var payload = message.getPayload();
        sendMessage(session, payload);
    }

    /** 发送错误
     * @param session 回话
     */
    @Override
    public void handleTransportError(WebSocketSession session, Throwable exception) {
        log.error("【websocket】会话错误", exception);
    }

    /**
     * 连接关闭
     */
    @Override
    public void afterConnectionClosed(WebSocketSession session, CloseStatus closeStatus) {
        String username = (String) session.getAttributes().get("username");
        sessions.remove(username);
        log.info("【websocket】用户断开连接:" + username);
    }

    @Override
    public boolean supportsPartialMessages() {
        return false;
    }

    /**
     * 发送消息
     * @param message
     */
    public void sendAll(Object message) {
        for(WebSocketSession session : sessions.values()) {
            sendMessage(session, message);
        }
    }

    /**
     * 发送消息
     * @param username
     * @param message
     */
    public void sendOne(String username, Object message) {
        WebSocketSession session = sessions.get(username);
        this.sendMessage(session, message);
    }

    /**
     * 发送消息
     * @param usernames
     * @param message
     */
    public void sendMultiple(String[] usernames, Object message) {
        for(String username: usernames) {
            this.sendOne(username, message);
        }
    }

}
