/**
 * 项目：互联网医疗
 * 模型分组：服务管理
 * 模型名称：订单表
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.rest;

import java.util.List;
import java.util.Arrays;

import cn.dev33.satoken.stp.SaLoginModel;
import cn.dev33.satoken.stp.StpUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import com.xwd.hospital.server.annotation.ApiCommon;
import com.xwd.hospital.server.annotation.AuthUser;
import com.xwd.hospital.server.base.BaseController;
import com.xwd.hospital.server.domain.User;
import com.xwd.hospital.server.dto.*;
import com.xwd.hospital.server.enums.OrderStateEnum;
import com.xwd.hospital.server.enums.OrderTypeEnum;
import com.xwd.hospital.server.rest.res.ApiCode;
import com.xwd.hospital.server.domain.Order;
import com.xwd.hospital.server.rest.req.OrderParam;
import com.xwd.hospital.server.rest.res.ApiResponse;
import com.xwd.hospital.server.service.OrderService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springdoc.core.annotations.ParameterObject;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@RestController
@Tag(name = "OrderController", description = "订单表管理")
@RequestMapping("/v1/pms/order")
public class OrderController extends BaseController<OrderService, Order, OrderParam> {

    @Resource
    public void setService(OrderService service) {
        this.service = service;
    }

    /**
     * 新增
     *
     * @param entity 要新增的对象
     * @return 新增的对象
     */
    @PostMapping("/")
    @Operation(summary = "新增")
    @Override
    public ApiResponse<Order> add(@RequestBody Order entity, @AuthUser User user) {
        entity.setEditorId(user.getId());
        entity.setEditorName(user.getUsername());
        entity.setDefault().validate(true);
        this.service.save(entity);
        return ApiResponse.ok(this.service.getById(entity.getId()));
    }

    /**
     * 批量新增
     *
     * @param entityList 要新增的对象
     * @return boolean 成功或失败
     */
    @PostMapping("/batch-save")
    @Operation(summary = "批量新增")
    @Override
    public ApiResponse<Boolean> batchSave(@RequestBody List<Order> entityList, @AuthUser User user) {
        entityList.forEach(entity -> {
            entity.setEditorId(user.getId());
            entity.setEditorName(user.getUsername());
            entity.setDefault().validate(true);
        });
        return ApiResponse.ok(this.service.saveBatch(entityList));
    }

    @Override
    @GetMapping("/{id}")
    @Operation(summary = "通过主键查询")
    @ApiCommon
    public ApiResponse selectByPrimaryKey(@PathVariable("id") Long id) {

        return ApiResponse.ok(this.service.queryOrderById(id));
    }

    /**
     * 小程序端订单列表
     *
     */
    @GetMapping("/queryOrderListForMiniApp")
    @Operation(summary = "小程序端订单列表")
    public ApiResponse queryOrderListForMiniApp(@RequestParam(value = "status",required = false)OrderStateEnum status,
                                                @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize) {
        Page<OrderDto> page = new Page<>(pageNum, pageSize);
//        List<OrderDto> orderDtoList = this.service.queryOrderListForMiniApp(status);
//        page.setRecords(orderDtoList);
        return ApiResponse.ok(this.service.queryOrderListForMiniApp(page,status));
    }

    /**
     * 患者端创建订单
     *
     */
    @PostMapping("/createOrder")
    @Operation(summary = "创建订单")
    @ApiCommon
    public ApiResponse<Order> createOrder(@RequestBody CreateOrderDto createOrderDto, @AuthUser User user) {
        Order order = this.service.createOrder(createOrderDto);
        if(null == order){
            return ApiResponse.fail(ApiCode.ORDER_CREATE_FAIL);

        }else {
            return ApiResponse.ok(order);
        }
    }

    /**
     * 医生端驳回订单
     *
     */
    @PostMapping("/rejectOrder")
    @Operation(summary = "医生端驳回订单")
    @ApiCommon
    public ApiResponse<Order> rejectOrder(@RequestBody RejectOrderDto rejectOrderDto, @AuthUser User user) {
        return ApiResponse.ok( this.service.rejectOrder(rejectOrderDto));
    }

    /**
     * 医生端接单（确认订单）
     *
     */
    @PostMapping("/acceptOrder")
    @Operation(summary = "医生端接单")
    @ApiCommon
    public ApiResponse<Order> acceptOrder(@RequestParam String orderNo, @AuthUser User user) {
        return ApiResponse.ok( this.service.acceptOrder(orderNo));
    }

    /**
     * 患者端发起聊天
     *
     */
    @PostMapping("/stratImChat")
    @Operation(summary = "患者端发起聊天")
    @ApiCommon
    public ApiResponse<Order> stratImChat(@RequestParam String orderNo, @AuthUser User user) {
        return ApiResponse.ok( this.service.stratImChat(orderNo));
    }

    /**
     * 患者端发起通话
     *
     */
    @PostMapping("/callPhone")
    @Operation(summary = "患者端发起通话")
    @ApiCommon
    public ApiResponse callPhone(@RequestParam String orderNo, @AuthUser User user) {
        return ApiResponse.ok( this.service.callPhone(orderNo));
    }

    /**
     * 医生端医生提交处方资料
     *
     */
    @PostMapping("/submitPrescriptionInfo")
    @Operation(summary = "医生端医生提交处方资料")
    @ApiCommon
    public ApiResponse<Order> submitPrescriptionInfo(@RequestBody PrescriptionInfoDto prescriptionInfoDto, @AuthUser User user) {
        return ApiResponse.ok( this.service.submitPrescriptionInfo(prescriptionInfoDto));
    }

    /**
     * 订单确认完成
     *
     */
    @PostMapping("/completeOrder")
    @Operation(summary = "订单确认完成")
    @ApiCommon
    public ApiResponse<Order> completeOrder(@RequestParam String orderNo, @AuthUser User user) {
        return ApiResponse.ok( this.service.completeOrder(orderNo));
    }

    /**
     * 订单评价
     *
     */
    @PostMapping("/appraiseOrder")
    @Operation(summary = "订单评价")
    @ApiCommon
    public ApiResponse<Order> appraiseOrder(@RequestBody AppraiseDto appraiseDto, @AuthUser User user) {
        return ApiResponse.ok( this.service.appraiseOrder(appraiseDto));
    }

    /**
     * 患者取消订单
     *
     */
    @PostMapping("/cancelOrder")
    @Operation(summary = "患者取消订单")
    @ApiCommon
    public ApiResponse<Order> cancelOrder(@RequestParam String orderNo, @AuthUser User user) {
        return ApiResponse.ok( this.service.cancelOrder(orderNo));
    }
}
