package com.xwd.hospital.server.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.xwd.hospital.server.domain.Order;
import com.xwd.hospital.server.domain.UserCouponInfo;
import com.xwd.hospital.server.enums.CouponUseStateEnum;
import com.xwd.hospital.server.enums.OrderStateEnum;
import com.xwd.hospital.server.repository.OrderMapper;
import com.xwd.hospital.server.service.OrderService;
import com.xwd.hospital.server.service.UserCouponInfoService;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Set;

@Service
@Slf4j
public class DelayQueueService {
    @Autowired
    private RedisTemplate<String,String> redisTemplate;

    @Resource
    private OrderMapper orderMapper;

    @Resource
    private UserCouponInfoService userCouponInfoService;

    private static final String DELAY_QUEUE_KEY = "order_delay_queue";

    public void addToDelayQueue(String message, long delayTime) {
        redisTemplate.opsForZSet().add(DELAY_QUEUE_KEY, message, System.currentTimeMillis() + delayTime);
    }

    public void pollAndProcessDelayedMessages() {
        Set<String> messages = redisTemplate.opsForZSet().rangeByScore(DELAY_QUEUE_KEY, 0, System.currentTimeMillis());
        for (String message : messages) {
            // 处理消息
            processMessage(message);
            // 从延迟队列中删除已处理的消息
            redisTemplate.opsForZSet().remove(DELAY_QUEUE_KEY, message);
        }
    }

    private void processMessage(String message) {
        // 未支付的订单取消
        //message就是orderNo
        //先查询订单状态
        Order order = orderMapper.selectOne(Wrappers.lambdaQuery(Order.class).eq(Order::getOrderNo, message));
        if (order == null) {
            log.info("订单不存在: " + message);
            return;
        }
        //判断订单是否是未支付状态
        if (!order.getOrderState().equals(OrderStateEnum.TO_PAY)) {
            log.info("订单已支付: " + message);
            return;
        }
        //修改订单状态为取消
        order.setOrderState(OrderStateEnum.CANCELED);
        orderMapper.updateById(order);

        //优惠券是否返回
        //查询优惠券
        if(null != order.getUserCouponId()){
            UserCouponInfo userCouponInfo = userCouponInfoService.getById(order.getUserCouponId());
            Date currentDate = new Date();

            if(currentDate.after(userCouponInfo.getCouponStartDate())
                    && currentDate.before(userCouponInfo.getCouponEndDate())){
                userCouponInfo.setCouponUseState(CouponUseStateEnum.USEABLE);
                userCouponInfo.setEditorId(1L);
                userCouponInfoService.saveOrUpdate(userCouponInfo);
            }
        }

        log.info("处理超时未支付订单------------: " + message);
    }
}
