/**
 * 项目：互联网医疗
 * 模型分组：服务管理
 * 模型名称：影像报告表
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.service.impl;

import java.io.Serializable;
import java.util.List;
import java.util.ArrayList;
import java.util.Collection;

import cn.dev33.satoken.stp.StpUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.IdcardUtil;
import com.xwd.hospital.server.domain.*;
import com.xwd.hospital.server.dto.ReportDto;
import com.xwd.hospital.server.enums.PatientRelationEnum;
import com.xwd.hospital.server.enums.ReportStateEnum;
import com.xwd.hospital.server.enums.YesNoEnum;
import com.xwd.hospital.server.service.*;
import jakarta.annotation.Resource;
import com.baomidou.mybatisplus.core.enums.SqlMethod;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.baomidou.mybatisplus.extension.toolkit.SqlHelper;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import com.xwd.hospital.server.repository.MedicalReportMapper;
import com.xwd.hospital.server.repository.base.MedicalReportBaseMapper;
import com.xwd.hospital.server.rest.req.MedicalReportParam;

@Service
public class MedicalReportServiceImpl extends ServiceImpl<MedicalReportMapper, MedicalReport> implements MedicalReportService {

    @Resource
    private MedicalReportPictureService pictureService;
    @Resource
    private UserInfoService userInfoService;
    @Resource
    private UserService userService;
    @Resource
    private PatientInfoService patientInfoService;

    @Override
    public int updateAllFieldsById(MedicalReport entity) {
        return this.getBaseMapper().updateAllFieldsById(entity);
    }

    /**
     * 批量插入
     *
     * @param entityList ignore
     * @param batchSize  ignore
     * @return ignore
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveBatch(Collection<MedicalReport> entityList, int batchSize) {
        String sqlStatement = SqlHelper.getSqlStatement(MedicalReportBaseMapper.class, SqlMethod.INSERT_ONE);
        return executeBatch(entityList, batchSize, (sqlSession, entity) -> sqlSession.insert(sqlStatement, entity));
    }

    @Override
    public boolean uploadReport(ReportDto dto) {
        //验证身份证号码
        boolean valid = IdcardUtil.isValidCard(dto.getPatientIdNo());
        if(!valid){
            return false;
        }

        //现根据身份证号查询是否有患者信息，如果没有则新增患者信息
        UserInfo userInfo = userInfoService.getOne(Wrappers.<UserInfo>query().eq("id_no", dto.getPatientIdNo()));
        if(null == userInfo){
            //新增患者信息
            userInfo = new UserInfo();
            userInfo.setUserId(0L);//关联sys_user表的id，暂时记为0
            userInfo.setName(dto.getPatientName());
            userInfo.setSex(dto.getSex());
            userInfo.setIdNo(dto.getPatientIdNo());
            userInfo.setPhoneNumber(dto.getPhoneNumber());

            DateTime birthday = DateUtil.parse(IdcardUtil.getBirthByIdCard(dto.getPatientIdNo()));
            userInfo.setBornDate(DateUtil.format(birthday,"yyyy-MM-dd"));
            userInfo.setPersonalInfoFilled(YesNoEnum.NO);
            userInfoService.save(userInfo);

            PatientInfo patientInfo = patientInfoService.getOne(Wrappers.<PatientInfo>query().eq("user_info_id", userInfo.getId())
                    .eq("relation", PatientRelationEnum.SELF).eq("id_no",dto.getPatientIdNo()));
            if(null == patientInfo){
                //同时创建自身的就诊人
                patientInfo = new PatientInfo();
                patientInfo.setUserInfoId(userInfo.getId());
                patientInfo.setRelation(PatientRelationEnum.SELF);
                patientInfo.setPatientName(userInfo.getName());
                patientInfo.setSex(userInfo.getSex());
                patientInfo.setIdNo(userInfo.getIdNo());
                patientInfo.setPhoneNumber(userInfo.getPhoneNumber());
                patientInfo.setBornDate(userInfo.getBornDate());
                patientInfoService.save(patientInfo);
            }
        }


        MedicalReport medicalReport = new MedicalReport();
        medicalReport.setAge(dto.getAge());
        medicalReport.setPatientName(dto.getPatientName());
        medicalReport.setReportDoctor(dto.getReportDoctor());
        medicalReport.setReportHospital(dto.getReportHospital());
        medicalReport.setPatientIdNo(dto.getPatientIdNo());
        medicalReport.setSex(dto.getSex());
        medicalReport.setReportItem(dto.getReportItem());
        medicalReport.setReportDeclare(dto.getReportDeclare());
        medicalReport.setReportTime(dto.getReportTime());
        this.save(medicalReport);

        if(null != dto.getPictureList()
            && dto.getPictureList().size()>0){
            ArrayList<MedicalReportPicture> pictureList = new ArrayList<>();
            for (String picUrl:dto.getPictureList()) {
                MedicalReportPicture picture = new MedicalReportPicture();
                picture.setMedicalReportId(medicalReport.getId());
                picture.setReportPicture(picUrl);
                pictureList.add(picture);
            }
            pictureService.saveBatch(pictureList);
        }

        return true;
    }

    @Override
    public MedicalReport myDoctorAsk() {
        //当前用户
        Object userId = StpUtil.getExtra("userId");
        UserInfo userInfo = userInfoService.getOne(Wrappers.<UserInfo>query().eq("user_id", Long.valueOf(userId.toString())));
        if(null == userInfo){
            return null;
        }
        User sysUser = userService.getById(Long.valueOf(userId.toString()));

        //根据medical_report中的身份证号查询和报告状态查询，查询报告状态为INTERPRETED下的最新一条医嘱，如果没有数据则查询报告状态为CONFIRMED下的最新一条医嘱
        MedicalReport medicalReport = this.getOne(Wrappers.<MedicalReport>query()
                .eq("patient_id_no", userInfo.getIdNo())
                .and(i -> i.eq("report_state", ReportStateEnum.INTERPRETED).or().eq("report_state", ReportStateEnum.CONFIRMED))
                .orderByDesc("report_state").orderByDesc("create_time").last("limit 1"));
        return medicalReport;
    }

    public static void main(String[] args) {
        DateTime birthday = DateUtil.parse(IdcardUtil.getBirthByIdCard("321088199311287715"));
        String format = DateUtil.format(birthday, "yyyy-MM-dd");
        System.out.println(format);
    }
}
