/**
 * 项目：互联网医疗
 * 模型分组：服务管理
 * 模型名称：订单表
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.service.impl;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

import cn.dev33.satoken.stp.StpUtil;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.xwd.hospital.server.constants.Constant;
import com.xwd.hospital.server.domain.*;
import com.xwd.hospital.server.dto.*;
import com.xwd.hospital.server.enums.*;
import com.xwd.hospital.server.service.*;
import jakarta.annotation.Resource;
import com.baomidou.mybatisplus.core.enums.SqlMethod;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.baomidou.mybatisplus.extension.toolkit.SqlHelper;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import com.xwd.hospital.server.repository.OrderMapper;
import com.xwd.hospital.server.repository.base.OrderBaseMapper;

@Service
public class OrderServiceImpl extends ServiceImpl<OrderMapper, Order> implements OrderService {
    @Resource
    private UserInfoService userInfoService;
    @Resource
    private UserService userService;
    @Resource
    private DoctorServiceInfoService doctorServiceInfoService;
    @Resource
    private UserCouponInfoService userCouponInfoService;
    @Resource
    private DoctorInfoService doctorInfoService;
    @Resource
    private PatientInfoService patientInfoService;
    @Resource
    private OrderAppraiseService orderAppraiseService;
    @Resource
    private PrescriptionOrderRecordService prescriptionOrderRecordService;
    @Resource
    private OrderOperationService orderOperationService;
    @Resource
    private OrderPictureService orderPictureService;
    @Resource
    private SmsService smsService;
    @Resource
    private PhoneOrderRecordService phoneOrderRecordService;
    @Resource
    private SettingService settingService;
    @Resource
    private OrderPayService orderPayService;
    @Resource
    private PaymentService paymentService;
    @Resource
    private DelayQueueService delayQueueService;

    /**
     * 计数count
     */
    private Integer count = 0;

    @Override
    public int updateAllFieldsById(Order entity) {
        return this.getBaseMapper().updateAllFieldsById(entity);
    }

    /**
     * 批量插入
     *
     * @param entityList ignore
     * @param batchSize  ignore
     * @return ignore
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveBatch(Collection<Order> entityList, int batchSize) {
        String sqlStatement = SqlHelper.getSqlStatement(OrderBaseMapper.class, SqlMethod.INSERT_ONE);
        return executeBatch(entityList, batchSize, (sqlSession, entity) -> sqlSession.insert(sqlStatement, entity));
    }

    @Override
    public OrderDetailDto queryOrderById(Long id) {
        OrderDetailDto orderDetailDto = new OrderDetailDto();
        Order order = this.getBaseMapper().selectById(id);
        BeanUtils.copyProperties(order,orderDetailDto);
        if(orderDetailDto.getOrderType() == OrderTypeEnum.PRESCRIPTION){
            PrescriptionOrderRecord record = prescriptionOrderRecordService.getOne(Wrappers.<PrescriptionOrderRecord>query().eq("order_id", order.getId()));
            if (null != record){
                orderDetailDto.setPrescriptionOrderRecord(record);
            }
        }
        //查询返回医生的聊天id
        DoctorInfo doctor = doctorInfoService.getById(orderDetailDto.getDoctorId());
        orderDetailDto.setImUserId(doctor.getUser().getImUserId());

        return orderDetailDto;
    }

    @Override
    public IPage<OrderDto> queryOrderListForMiniApp(Page page,OrderStateEnum param) {
        Object userId = StpUtil.getExtra("userId");
        UserInfo userInfo = userInfoService.getOne(Wrappers.<UserInfo>query().eq("user_id", Long.valueOf(userId.toString())));
        if(null == userInfo){
            return null;
        }
        return this.getBaseMapper().queryOrderListForMiniApp(page,userInfo.getId(),null == param?null:param.toString());
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public Order createOrder(CreateOrderDto createOrderDto) {
        //当前用户
        Object userId = StpUtil.getExtra("userId");
        UserInfo userInfo = userInfoService.getOne(Wrappers.<UserInfo>query().eq("user_id", Long.valueOf(userId.toString())));
        if(null == userInfo){
            return null;
        }
        User sysUser = userService.getById(Long.valueOf(userId.toString()));

        Order order = new Order();
        //写入订单相关记录记录
        order.setUserInfoId(userInfo.getId());
        order.setDoctorId(createOrderDto.getDoctorId());
        order.setPatientId(createOrderDto.getPatientId());
        //生成订单号
        //todo 订单号生成规则待定
        order.setOrderNo(geneartOrderNo());
        order.setOrderState(OrderStateEnum.TO_PAY);
        order.setOrderType(createOrderDto.getOrderType());
        order.setAppointmentDate(createOrderDto.getAppointmentDate());

        //查询相关服务价格
        OrderTypeEnum orderType = createOrderDto.getOrderType();
        Long doctorId = createOrderDto.getDoctorId();
        DoctorInfo doctorInfo = doctorInfoService.getById(doctorId);

        DoctorServiceInfo doctorServiceInfo = doctorServiceInfoService.getOne(
                Wrappers.<DoctorServiceInfo>query().eq("doctor_id", doctorId)
                        .eq("service_name", orderType.toString())
                        .eq("service_state", YesNoEnum.YES.toString()));
        if(null == doctorServiceInfo){
            //todo 抛出异常
            return null;
        }
        order.setOrderPrice(doctorServiceInfo.getServicePrice());

        //查询优惠金额
        order.setOrderDiscount(BigDecimal.ZERO);
        if(null != createOrderDto.getUserCouponId()){
            UserCouponInfo userCouponInfo = userCouponInfoService.getById(createOrderDto.getUserCouponId());
            //校验优惠券有效性
            if(checkCoupon(userCouponInfo)){
                order.setOrderDiscount(userCouponInfo.getCouponDiscount());
                order.setUserCouponId(createOrderDto.getUserCouponId());

                //优惠券变为已使用
                userCouponInfo.setCouponUseState(CouponUseStateEnum.USED);
                userCouponInfo.setEditorId(sysUser.getId());
                userCouponInfo.setEditorName(sysUser.getUsername());
                userCouponInfoService.saveOrUpdate(userCouponInfo);
            }
        }

        //计算实际价格
        BigDecimal caulatePrice = order.getOrderPrice().subtract(order.getOrderDiscount());
        order.setOrderActualPay(caulatePrice.compareTo(BigDecimal.ZERO) > 0?order.getOrderPrice().subtract(order.getOrderDiscount()) : BigDecimal.ZERO );
        if(0 == order.getOrderActualPay().compareTo(BigDecimal.ZERO)){
            //0元订单直接
            order.setOrderState(OrderStateEnum.TO_CONFIRM);
            order.setPayTime(new Date());

            OrderPay orderPay = new OrderPay();
            orderPay.setOrderId(order.getId());
            orderPay.setPayNo(geneartOrderNo());
            orderPay.setOrderNo(order.getOrderNo());
            orderPay.setPayState(PayStateEnum.PAYED);
            orderPay.setPayPath(PayPathEnum.WX_PAY);
            orderPay.setPayMoney(order.getOrderActualPay());
            orderPay.setEditorId(StpUtil.getLoginIdAsLong());
            orderPay.setPayTime(new Date());
            orderPayService.save(orderPay);
        }
        //写入服务相关信息
        order.setServiceStartTime(doctorServiceInfo.getServiceStartTime());
        order.setServiceEndTime(doctorServiceInfo.getServiceEndTime());
        order.setDoctorName(doctorInfo.getDoctorName());
        order.setHospitalName(doctorInfo.getMedicalQualification().getHospital().getHospitalName());
        order.setDepartmentName(doctorInfo.getMedicalQualification().getDepartment().getDepartmentName());

        //写入就诊人相关信息
        PatientInfo patientInfo = patientInfoService.getById(createOrderDto.getPatientId());
        order.setPatientName(patientInfo.getPatientName());
        order.setPatientSex(patientInfo.getSex());
        order.setPatientIdNo(patientInfo.getIdNo());
        //如果存在监护人，则手机号码填写监护人的手机号码
        if(patientInfo.getGuardianState() == YesNoEnum.YES){
            order.setPatientPhoneNumber(patientInfo.getGuardianPhoneNumber());
        }else {
            order.setPatientPhoneNumber(patientInfo.getPhoneNumber());
        }
        order.setPatientBornDate(patientInfo.getBornDate());
        order.setPatientSelfIntroduce(createOrderDto.getPatientSelfIntroduce());
        order.setEditorId(sysUser.getId());
        order.setEditorName(sysUser.getUsername());
        this.save(order);

        //写入患者自述相关图片
        if(null != createOrderDto.getPictureList()
            && createOrderDto.getPictureList().size()>0){
            ArrayList<OrderPicture> orderPictures = new ArrayList<>();
            for (String picUrl:createOrderDto.getPictureList()) {
                OrderPicture orderPicture = new OrderPicture();
                orderPicture.setOrderId(order.getId());
                orderPicture.setOrderNo(order.getOrderNo());
                orderPicture.setPatientSelfImage(picUrl);
                orderPicture.setEditorId(sysUser.getId());
                orderPicture.setEditorName(sysUser.getUsername());
                orderPictures.add(orderPicture);
            }
            orderPictureService.saveBatch(orderPictures);
        }

        //todo 根据订单类型，写入相关记录表
        if(order.getOrderType() == OrderTypeEnum.PRESCRIPTION){
            //续方开药订单
            PrescriptionOrderRecord prescriptionOrderRecord = new PrescriptionOrderRecord();
            prescriptionOrderRecord.setPrescriptionId(createOrderDto.getPatientPrescriptionId());
            prescriptionOrderRecord.setOrderId(order.getId());
            prescriptionOrderRecord.setOrderNo(order.getOrderNo());
            prescriptionOrderRecordService.save(prescriptionOrderRecord);

        }else if(order.getOrderType() == OrderTypeEnum.TELEPHONE){
            PhoneOrderRecord phoneOrderRecord = new PhoneOrderRecord();
            phoneOrderRecord.setOrderId(order.getId());
            phoneOrderRecord.setOrderNo(order.getOrderNo());
            phoneOrderRecord.setPatientPhoneNo(order.getPatientPhoneNumber());
            phoneOrderRecordService.save(phoneOrderRecord);
        }


        //写入订单操作记录
        OrderOperation orderOperation = new OrderOperation();
        orderOperation.setOrderId(order.getId());
        orderOperation.setOrderNo(order.getOrderNo());
        orderOperation.setOperateInfo(OrderOperationEnum.ORDER_CREATED.getName());
        orderOperation.setEditorId(sysUser.getId());
        orderOperation.setEditorName(sysUser.getUsername());
        orderOperationService.save(orderOperation);

        //写入支付延时队列
        delayQueueService.addToDelayQueue(order.getOrderNo(), Constant.ORDER_PAY_WAIT_TIME);

        return order;
    }

    /**
     * 生成订单
     * @return
     */
    public static String geneartOrderNo() {
        //生成方式：当前时间+随机6位数字
        // 获取当前日期时间
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
        String dateTime = dateFormat.format(new Date());

        // 生成6位随机数
        Random random = new Random();
        int randomValue = random.nextInt(900000) + 100000; // 生成100000到999999之间的随机数

        // 将时间戳和随机数拼接成订单号
        String serialNumber = String.valueOf(dateTime) + String.valueOf(randomValue);

        return serialNumber;
    }

    /**
     * 校验优惠券有效性
     * @return
     */
    public static boolean checkCoupon(UserCouponInfo userCouponInfo) {
        //校验优惠券可用状态
        if(userCouponInfo.getCouponUseState() != CouponUseStateEnum.USEABLE){
            return false;
        }
        //校验优惠券有效日期
        Date today = new Date();

        if(today.after(userCouponInfo.getCouponStartDate()) && today.before(userCouponInfo.getCouponEndDate())){
            return true;
        }else {
            return false;
        }
    }

    /**
     * 取消订单
     * @param orderNo
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public Order cancelOrder(String orderNo) {
        //修改订单状态
        Order order = this.getOne(Wrappers.<Order>query().eq("order_no", orderNo));
        if(order.getOrderState() == OrderStateEnum.TO_PAY){//待支付的订单方可取消
            order.setOrderState(OrderStateEnum.CANCELED);
            this.saveOrUpdate(order);

            //优惠券是否返回
            //查询优惠券
            if(null != order.getUserCouponId()){
                UserCouponInfo userCouponInfo = userCouponInfoService.getById(order.getUserCouponId());
                Date currentDate = new Date();

                if(currentDate.after(userCouponInfo.getCouponStartDate())
                        && currentDate.before(userCouponInfo.getCouponEndDate())){
                    userCouponInfo.setCouponUseState(CouponUseStateEnum.USEABLE);
                    userCouponInfo.setEditorId(1L);
                    userCouponInfoService.saveOrUpdate(userCouponInfo);
                }
            }
        }else if(order.getOrderState() == OrderStateEnum.TO_CONFIRM){//待确认订单
            order.setOrderState(OrderStateEnum.CANCELED);
            this.saveOrUpdate(order);

            //优惠券是否返回
            //查询优惠券
            if(null != order.getUserCouponId()){
                UserCouponInfo userCouponInfo = userCouponInfoService.getById(order.getUserCouponId());
                Date currentDate = new Date();

                if(currentDate.after(userCouponInfo.getCouponStartDate())
                        && currentDate.before(userCouponInfo.getCouponEndDate())){
                    userCouponInfo.setCouponUseState(CouponUseStateEnum.USEABLE);
                    userCouponInfo.setEditorId(1L);
                    userCouponInfoService.saveOrUpdate(userCouponInfo);
                }
            }

            //退款操作
            //todo 全额退款
            paymentService.refundPay(order.getOrderNo());

        }

        return order;
    }

    /**
     * 订单驳回
     *
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public Order rejectOrder(RejectOrderDto rejectOrderDto) {
        //修改订单状态
        Order order = this.getOne(Wrappers.<Order>query().eq("order_no", rejectOrderDto.getOrderNo()));
        if(order.getOrderState() == OrderStateEnum.TO_CONFIRM){
            order.setOrderState(OrderStateEnum.REJECTED);
            order.setCancelReason(rejectOrderDto.getReason());
            this.saveOrUpdate(order);

            //优惠券是否返回
            if(null != order.getUserCouponId()){
                //查询是否使用优惠券优惠券
                UserCouponInfo userCouponInfo = userCouponInfoService.getById(order.getUserCouponId());
                Date currentDate = new Date();
                //如果使用了优惠券，并且仍符合使用规则
                if(currentDate.after(userCouponInfo.getCouponStartDate())
                        && currentDate.before(userCouponInfo.getCouponEndDate())){
                    //优惠券状态设置为可用
                    userCouponInfo.setCouponUseState(CouponUseStateEnum.USEABLE);
                    userCouponInfo.setEditorId(1L);
                    userCouponInfoService.saveOrUpdate(userCouponInfo);
                }
            }
        }
        //todo 全额退款
        paymentService.refundPay(order.getOrderNo());
        return order;
    }

    /**
     * 医生确认接单
     * @param orderNo
     * @return
     */
    @Override
    public Order acceptOrder(String orderNo) {
        //修改订单状态
        Order order = this.getOne(Wrappers.<Order>query().eq("order_no", orderNo));
        if(order.getOrderState() == OrderStateEnum.TO_CONFIRM){
            order.setOrderState(OrderStateEnum.IN_SERVICE);
            order.setConfirmTime(new Date());
            this.saveOrUpdate(order);
        }

        return order;
    }

    /**
     * 开启IM在线聊天
     * @param orderNo
     * @return
     */
    @Override
    public Order stratImChat(String orderNo) {
        //todo 接入IM
        return null;
    }

    /**
     * 开启匿名通话
     * @param orderNo
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public String callPhone(String orderNo) {
        String proxyNumber = null;
        //判断订单状态是否为服务中
        Order order = this.getOne(Wrappers.<Order>query().eq("order_no", orderNo));
        if(order.getOrderState() == OrderStateEnum.IN_SERVICE){
            //判断预约日期是否到达
            Date now = new Date();
            Date appointmentDate = order.getAppointmentDate();
            int compare = DateUtil.compare(now, appointmentDate, "yyyy-MM-dd");
            if(0 != compare){
                return null;
            }
            //绑定小号和患者号码并返回小号
            PhoneOrderRecord record = phoneOrderRecordService.getOne(Wrappers.<PhoneOrderRecord>query().eq("order_no", orderNo));
//            if(!StringUtils.isBlank(record.getProxyPhoneNo() )
//                &&record.getMappingState() == YesNoEnum.NO){
//                return record.getProxyPhoneNo();
//
//            }
            //todo 从小号列表中获取到合适的小号(sys_setting表)
            List<Setting> settingList = settingService.list(Wrappers.<Setting>query().eq("name", SysSettingEnum.PROXY_PHONE));
            if(null == settingList
                || settingList.size() == 0){
                //暂无可用来绑定的小号
                return null;
            }
            Setting setting = settingList.get(count);
            proxyNumber = setting.getValue();

            //绑定小号和患者号码
            String mappingId = smsService.bindPhone(proxyNumber, record.getPatientPhoneNo());
            record.setMappingId(mappingId);
            record.setProxyPhoneNo(proxyNumber);
            record.setMappingState(YesNoEnum.NO);
            phoneOrderRecordService.saveOrUpdate(record);
            //小号改为已使用
            setting.setRemark(UseStateEnum.USED.getName());
            settingService.saveOrUpdate(setting);

            if(Constant.PROXY_NUMBER_LIMIT == count){
                count = 0;
            }else {
                count++;
            }

        }

        return proxyNumber;
    }

    /**
     * 提交续方开药处方单
     * @param prescriptionInfoDto
     * @return
     */
    @Override
    public Order submitPrescriptionInfo(PrescriptionInfoDto prescriptionInfoDto) {
        //查询订单
        Order order = this.getOne(Wrappers.<Order>query().eq("order_no", prescriptionInfoDto.getOrderNo()));
        if(null !=order && order.getOrderState() == OrderStateEnum.IN_SERVICE){
            PrescriptionOrderRecord record = prescriptionOrderRecordService.getOne(Wrappers.<PrescriptionOrderRecord>query().eq("order_no", order.getOrderNo()));
            record.setPrescriptionImage(prescriptionInfoDto.getPrescriptionImage());
            record.setPrescriptionIntroduce(prescriptionInfoDto.getPrescriptionIntroduce());
            prescriptionOrderRecordService.saveOrUpdate(record);
        }

        return order;
    }

    /**
     * 确认订单完成
     * @param orderNo
     * @return
     */
    @Override
    public Order completeOrder(String orderNo) {
        //修改订单状态
        Order order = this.getOne(Wrappers.<Order>query().eq("order_no", orderNo));
        if(order.getOrderState() == OrderStateEnum.IN_SERVICE){
            order.setOrderState(OrderStateEnum.COMPLETED);
            order.setCompleteTime(new Date());
            this.saveOrUpdate(order);
        }
        return order;
    }

    /**
     * 订单评价
     * @param appraiseDto
     * @return
     */
    @Override
    public Order appraiseOrder(AppraiseDto appraiseDto) {
        //修改订单状态
        Order order = this.getOne(Wrappers.<Order>query().eq("order_no", appraiseDto.getOrderNo()));
        if(order.getOrderState() == OrderStateEnum.COMPLETED){
            //已完成订单才可以评价
            OrderAppraise orderAppraise = new OrderAppraise();
            orderAppraise.setOrderNo(appraiseDto.getOrderNo());
            orderAppraise.setOrderId(order.getId());
            orderAppraise.setOrderScore(appraiseDto.getOrderScore());
            orderAppraise.setOrderEvaluation(appraiseDto.getOrderEvaluation());
            orderAppraise.setEditorId(order.getEditorId());
            orderAppraiseService.save(orderAppraise);

        }
        return order;
    }
}
