/**
 * 项目：互联网医疗
 * 模型分组：系统管理
 * 模型名称：用户表
 * @Author: xiongwei
 * @Date: 2023-09-05 09:42:00
 */

package com.xwd.hospital.server.service.impl;

import java.io.IOException;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;

import cn.binarywang.wx.miniapp.api.WxMaService;
import cn.binarywang.wx.miniapp.bean.WxMaJscode2SessionResult;
import cn.binarywang.wx.miniapp.bean.WxMaPhoneNumberInfo;
import cn.binarywang.wx.miniapp.bean.WxMaUserInfo;
import cn.binarywang.wx.miniapp.util.WxMaConfigHolder;
import cn.dev33.satoken.stp.StpUtil;
import cn.hutool.core.lang.UUID;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.IdcardUtil;
import com.google.gson.JsonObject;
import com.tencentyun.TLSSigAPIv2;
import com.xwd.hospital.server.config.WxMaProperties;
import com.xwd.hospital.server.constants.Constant;
import com.xwd.hospital.server.domain.*;
import com.xwd.hospital.server.dto.LoginDto;
import com.xwd.hospital.server.dto.RegisterDto;
import com.xwd.hospital.server.dto.UpdateInfoDto;
import com.xwd.hospital.server.dto.UpdateUserInfoDto;
import com.xwd.hospital.server.dto.UserDto;
import com.xwd.hospital.server.enums.*;
import com.xwd.hospital.server.rest.thirdPartReq.BindIMUserReq;
import com.xwd.hospital.server.rest.thirdPartReq.BindNumberReq;
import com.xwd.hospital.server.rest.thirdPartReq.ProfileItem;
import com.xwd.hospital.server.rest.thirdPartReq.UpdateIMUserReq;
import com.xwd.hospital.server.service.*;
import jakarta.annotation.Resource;
import com.baomidou.mybatisplus.core.enums.SqlMethod;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.baomidou.mybatisplus.extension.toolkit.SqlHelper;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.error.WxErrorException;
import okhttp3.OkHttpClient;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import com.xwd.hospital.server.repository.UserMapper;
import com.xwd.hospital.server.repository.base.UserBaseMapper;
import com.xwd.hospital.server.rest.res.ApiCode;
import cn.dev33.satoken.secure.SaSecureUtil;
import com.xwd.hospital.server.repository.RoleMapper;
import com.xwd.hospital.server.repository.UserRoleMapper;
import retrofit2.Call;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;

@Service
@Slf4j
public class UserServiceImpl extends ServiceImpl<UserMapper, User> implements UserService {
    @Resource
    private UserRoleMapper userRoleMapper;
    @Resource
    private RoleMapper roleMapper;
    @Resource
    private WxMaService wxMaService;
    @Resource
    private WxMaProperties wxMaProperties;
    @Resource
    private RedisService redisService;
    @Resource
    private UserInfoService userInfoService;
    @Resource
    private DoctorInfoService doctorInfoService;
    @Resource
    private PatientInfoService patientInfoService;
    @Resource
    private DoctorServiceInfoService doctorServiceInfoService;
    @Resource
    private InviteRegisterRecordService inviteRegisterRecordService;

    @Value("${wx.miniapp.configs[0].appid}")
    private String patientAppId;

    @Value("${wx.miniapp.configs[1].appid}")
    private String doctorAppId;

    @Value("${tencent.IM.sdkAppId}")
    private Long sdkAppId;

    @Value("${tencent.IM.key}")
    private String key;

    @Value("${tencent.IM.expire}")
    private Long expire;

    @Value("${tencent.IM.host}")
    public String imHost;

    @Override
    public int updateAllFieldsById(User entity) {
        return this.getBaseMapper().updateAllFieldsById(entity);
    }

    /**
     * 批量插入
     *
     * @param entityList ignore
     * @param batchSize  ignore
     * @return ignore
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveBatch(Collection<User> entityList, int batchSize) {
        String sqlStatement = SqlHelper.getSqlStatement(UserBaseMapper.class, SqlMethod.INSERT_ONE);
        return executeBatch(entityList, batchSize, (sqlSession, entity) -> sqlSession.insert(sqlStatement, entity));
    }

    @Override
    public User login(String username, String password) {
        User user = this.selectByUsername(username);
        if (user == null) {
            throw new ApiCode.ApiException(-1, "用户名不存在！");
        }
        if (!SaSecureUtil.sha256(password).equals(user.getPassword())) {
            throw new ApiCode.ApiException(-3, "用户名和密码不匹配！");
        }
        return user;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public User register(User user, String roleName) {
        user.setDefault().validate(true);
        if (user.getPassword() == null) {
            throw new ApiCode.ApiException(-1, "用户密码不能为空！");
        }
        user.setPassword(SaSecureUtil.sha256(user.getPassword()));
        // 判断用户名是否存在
        User existUser = this.selectByUsername(user.getUsername());
        if (existUser != null) {
            throw new ApiCode.ApiException(-2, "用户已注册，请登录！");
        }
        this.save(user);
        // 分配默认角色给用户
        Role defaultRole = this.roleMapper.selectOne(Wrappers.<Role>lambdaQuery().eq(Role::getName, roleName));
        if (defaultRole != null) {
            this.userRoleMapper.insert(UserRole.newBuilder().userId(user.getId()).roleId(defaultRole.getId()).build());
        }
        return this.getById(user.getId());
    }

    @Override
    public User selectByUsername(String username) {
        return this.getOne(Wrappers.<User>query().eq("username", username));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean save(User entity) {
        boolean result = super.save(entity);
        if (entity.getRoleList() != null) {
            this.userRoleMapper.delete(Wrappers.<UserRole>query().eq("user_id", entity.getId()));
            List<UserRole> userRoleList = new ArrayList<>();
            for (Role role: entity.getRoleList()) {
                UserRole userRole = new UserRole();
                userRole.setUserId(entity.getId());
                userRole.setRoleId(role.getId());
                userRoleList.add(userRole);
            }
            if (userRoleList.size() > 0) {
                this.userRoleMapper.batchInsert(userRoleList);
            }
        }
        return result;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateById(User entity) {
        boolean result = super.updateById(entity);
        if (entity.getRoleList() != null) {
            this.userRoleMapper.delete(Wrappers.<UserRole>query().eq("user_id", entity.getId()));
            List<UserRole> userRoleList = new ArrayList<>();
            for (Role role: entity.getRoleList()) {
                UserRole userRole = new UserRole();
                userRole.setUserId(entity.getId());
                userRole.setRoleId(role.getId());
                userRoleList.add(userRole);
            }
            if (userRoleList.size() > 0) {
                this.userRoleMapper.batchInsert(userRoleList);
            }
        }
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public UserDto miniAppLogin(LoginDto loginDto){
        UserDto userDto = new UserDto();
        //短信验证码登录
        if (StringUtils.isBlank(loginDto.getCode())) {
            String authCode = redisService.get(Constant.SMS_PREFIX + loginDto.getPhoneNumber());
            if(StringUtils.isBlank(authCode) ||
                    StringUtils.isBlank(loginDto.getSmsAuthCode()) ||
                    !authCode.equals(loginDto.getSmsAuthCode())){
                //短信验证码校验不通过,直接返回空userDto
                userDto.setAuthFlag(false);
                return userDto;
            }else {
                userDto.setAuthFlag(true);
                User user = this.getOne(Wrappers.<User>query().eq("phone", loginDto.getPhoneNumber()).eq("user_type",loginDto.getUserType()));
                if(null == user){
                    //此手机号尚未在系统中注册，直接注册为新用户
                    user = new User();
                    if(loginDto.getUserType() == UserTypeEnum.PATIENT){
                        user.setUsername(UserTypeEnum.PATIENT.getName()+"_"+loginDto.getPhoneNumber());
                    }else if(loginDto.getUserType() == UserTypeEnum.DOCTOR){
                        user.setUsername(UserTypeEnum.DOCTOR.getName()+"_"+loginDto.getPhoneNumber());
                    }
                    user.setUserType(loginDto.getUserType());
                    user.setRegisterSource(RegisterSourceEnum.WX);
                    user.setPhone(loginDto.getPhoneNumber());
                    user.setImUserId(IdUtil.getSnowflakeNextIdStr());
                    //添加用户信息到IM系统
                    addUserToIMSystem(user.getImUserId(),user.getUsername(),user.getAvatar());

                    super.save(user);
                    //分配角色
                    String roleName = "";
                    if(loginDto.getUserType() == UserTypeEnum.PATIENT){
                        roleName = RoleNameEnum.ROLE_PATIENT.toString();
                        //同时生成相关信息表
                        UserInfo userInfo = new UserInfo();
                        userInfo.setUserId(user.getId());
                        userInfo.setName(user.getUsername());
                        userInfo.setPersonalInfoFilled(YesNoEnum.NO);
                        userInfoService.save(userInfo);
                        userDto.setUserInfo(userInfo);
                        user.setObjId(userInfo.getId());
                    }else if(loginDto.getUserType() == UserTypeEnum.DOCTOR){
                        roleName = RoleNameEnum.ROLE_DOCTOR.toString();
                        //同时生成相关信息表
                        DoctorInfo doctorInfo = new DoctorInfo();
                        doctorInfo.setUserId(user.getId());
                        doctorInfo.setDoctorName(user.getUsername());
                        doctorInfoService.save(doctorInfo);
                        userDto.setDoctorInfo(doctorInfo);
                        user.setObjId(doctorInfo.getId());

                        createService(doctorInfo.getId());

                    }
                    super.saveOrUpdate(user);
                    Role defaultRole = this.roleMapper.selectOne(Wrappers.<Role>lambdaQuery().eq(Role::getName, roleName));

                    this.userRoleMapper.insert(UserRole.newBuilder().userId(user.getId()).roleId(defaultRole.getId()).build());
                    //检查是否存在邀请
                    if(!StringUtils.isEmpty(loginDto.getInviterId())){
                        //存在邀请注册
                        InviteRegisterRecord inviteRegisterRecord = new InviteRegisterRecord();
                        //邀请人
                        inviteRegisterRecord.setInviteUserId(Long.valueOf(loginDto.getInviterId()));
                        //被邀请人
                        inviteRegisterRecord.setInvitedUserId(user.getId());
                        inviteRegisterRecordService.save(inviteRegisterRecord);
                    }
                }else {
                    //已存在用户，直接登录
                    if(null != user.getObjId()){
                        //关联信息表
                        if(user.getUserType() == UserTypeEnum.PATIENT){
                            userDto.setUserInfo(userInfoService.getById(user.getObjId()));
                        }else if(user.getUserType() == UserTypeEnum.DOCTOR){
                            userDto.setDoctorInfo(doctorInfoService.getById(user.getObjId()));
                        }
                    }
                }

                userDto.setUser(user);
                return userDto;
            }
        }else {//微信直接授权登录
            //根据小程序登录源，切换相应配置
            if(loginDto.getUserType() == UserTypeEnum.PATIENT){
                wxMaService.switchover(patientAppId);
                log.debug(patientAppId);
            }else if(loginDto.getUserType() == UserTypeEnum.DOCTOR){
                wxMaService.switchover(doctorAppId);
                log.debug(doctorAppId);
            }
            try {
                String appid = wxMaProperties.getConfigs().get(0).getAppid();
                WxMaJscode2SessionResult session = wxMaService.getUserService().getSessionInfo(loginDto.getCode());
                log.debug(session.getSessionKey());
                log.debug(session.getOpenid());
                //通过openId与sys_user表关联 判断是否需要进行注册流程
                String openid = session.getOpenid();
                User user = this.getOne(Wrappers.<User>query().eq("open_id", openid).eq("user_type",loginDto.getUserType()));
                if(null == user){
                    //open_id未查询到相关用户,直接返回走注册流程
                    userDto.setRegisterFlag(true);
                    return userDto;
                }else{
                    //已存在用户，直接登录
                    userDto.setUser(user);
                    if(null != user.getObjId()){
                        //查询关联的信息表
                        if(user.getUserType() == UserTypeEnum.PATIENT){
                            userDto.setUserInfo(userInfoService.getById(user.getObjId()));
                        }else if(user.getUserType() == UserTypeEnum.DOCTOR){
                            userDto.setDoctorInfo(doctorInfoService.getById(user.getObjId()));
                        }
                    }
                    return userDto;
                }
            } catch (WxErrorException e) {
                log.error("微信登录异常");
            } finally {
                WxMaConfigHolder.remove();//清理ThreadLocal
            }
        }
        return userDto;
    };

    void createService(Long doctorId){
        ArrayList<DoctorServiceInfo> serviceList = new ArrayList<>(3);
        DoctorServiceEnum[] serviceEnums = {
                DoctorServiceEnum.ONLINE_INQUIRY,
                DoctorServiceEnum.PRESCRIPTION,
                DoctorServiceEnum.TELEPHONE
        };
        for (int i = 0; i < 3; i++) {
            DoctorServiceInfo doctorServiceInfo = new DoctorServiceInfo();
            doctorServiceInfo.setDoctorId(doctorId);
            doctorServiceInfo.setServiceState(YesNoEnum.NO);
            doctorServiceInfo.setServicePrice(new BigDecimal("100"));
            doctorServiceInfo.setServiceStartTime("00:00");
            doctorServiceInfo.setServiceEndTime("23:59");
            doctorServiceInfo.setServicePhoneNumber("");
            doctorServiceInfo.setServiceName(serviceEnums[i]);
            serviceList.add(doctorServiceInfo);
        }

        doctorServiceInfoService.saveBatch(serviceList);
    }

    @Override
    public UserDto miniAppRegister(RegisterDto registerDto) {
        UserDto userDto = new UserDto();
        //根据小程序登录源，切换相应配置
        if(registerDto.getUserType() == UserTypeEnum.PATIENT){
            wxMaService.switchover(patientAppId);
        }else if(registerDto.getUserType() == UserTypeEnum.DOCTOR){
            wxMaService.switchover(doctorAppId);
        }
        try {
            WxMaJscode2SessionResult session = wxMaService.getUserService().getSessionInfo(registerDto.getCode());
            log.debug(session.getSessionKey());
            log.debug(session.getOpenid());
            //通过openId与sys_user表关联 判断是否需要进行注册流程
            String openid = session.getOpenid();
            User user = this.getOne(Wrappers.<User>query().eq("phone", registerDto.getPhoneNumber()).eq("user_type",registerDto.getUserType()));
            if(null != user ){
                //此手机号已注册，直接绑定open_id
                user.setOpenId(openid);
                if(null != user.getObjId()){
                    //关联信息表
                    if(user.getUserType() == UserTypeEnum.PATIENT){
                        userDto.setUserInfo(userInfoService.getById(user.getObjId()));
                    }else if(user.getUserType() == UserTypeEnum.DOCTOR){
                        userDto.setDoctorInfo(doctorInfoService.getById(user.getObjId()));
                    }
                }
            }else {
                //尚未在系统中注册，直接注册为新用户
                user = new User();
                if(registerDto.getUserType() == UserTypeEnum.PATIENT){
                    user.setUsername(UserTypeEnum.PATIENT.getName()+"_"+registerDto.getPhoneNumber());
                }else if(registerDto.getUserType() == UserTypeEnum.DOCTOR){
                    user.setUsername(UserTypeEnum.DOCTOR.getName()+"_"+registerDto.getPhoneNumber());
                }
                user.setUserType(registerDto.getUserType());
                user.setRegisterSource(RegisterSourceEnum.WX);
                user.setPhone(registerDto.getPhoneNumber());
                user.setOpenId(openid);
                user.setImUserId(IdUtil.getSnowflakeNextIdStr());
                addUserToIMSystem(user.getImUserId(),user.getUsername(),user.getAvatar());
                this.saveOrUpdate(user);
                //分配用户角色role
                String roleName = "";
                if(registerDto.getUserType() == UserTypeEnum.PATIENT){
                    roleName = RoleNameEnum.ROLE_PATIENT.toString();
                    //同时生成相关信息表
                    UserInfo userInfo = new UserInfo();
                    userInfo.setUserId(user.getId());
                    userInfo.setName(user.getUsername());
                    userInfo.setPersonalInfoFilled(YesNoEnum.NO);
                    userInfoService.save(userInfo);
                    userDto.setUserInfo(userInfo);
                    user.setObjId(userInfo.getId());
                }else if(registerDto.getUserType() == UserTypeEnum.DOCTOR){
                    roleName = RoleNameEnum.ROLE_DOCTOR.toString();
                    //同时生成相关信息表
                    DoctorInfo doctorInfo = new DoctorInfo();
                    doctorInfo.setUserId(user.getId());
                    doctorInfo.setDoctorName(user.getUsername());
                    doctorInfoService.save(doctorInfo);
                    userDto.setDoctorInfo(doctorInfo);
                    user.setObjId(doctorInfo.getId());
                    createService(doctorInfo.getId());
                }
                Role defaultRole = this.roleMapper.selectOne(Wrappers.<Role>lambdaQuery().eq(Role::getName, roleName));

                this.userRoleMapper.insert(UserRole.newBuilder().userId(user.getId()).roleId(defaultRole.getId()).build());

                //检查是否存在邀请
                if(!StringUtils.isEmpty(registerDto.getInviterId())){
                    //存在邀请注册
                    InviteRegisterRecord inviteRegisterRecord = new InviteRegisterRecord();
                    //邀请人
                    inviteRegisterRecord.setInviteUserId(Long.valueOf(registerDto.getInviterId()));
                    //被邀请人
                    inviteRegisterRecord.setInvitedUserId(user.getId());
                    inviteRegisterRecordService.save(inviteRegisterRecord);
                }
            }
            userDto.setUser(user);
            this.saveOrUpdate(user);
            return userDto;
        } catch (WxErrorException e) {
            log.error("微信登录异常");
        } finally {
            WxMaConfigHolder.remove();//清理ThreadLocal
        }
        return userDto;
    }

    @Override
    public UserDto updateUserInfo(UpdateUserInfoDto updateUserDto) {
        boolean valid = IdcardUtil.isValidCard(updateUserDto.getIdNo());
        if(!valid){
            return null;
        }

        Long userId = Long.valueOf(StpUtil.getExtra("userId").toString());

        User user = this.getById(userId);

        UserDto userDto = new UserDto();

        //插入obj_id和相关信息记录
        if(updateUserDto.getUserType() == UserTypeEnum.PATIENT){
            UserInfo userInfo = userInfoService.getOne(Wrappers.<UserInfo>query().eq("user_id", userId));
            if(null == userInfo){
                UserInfo oldUserInfo = userInfoService.getOne(Wrappers.<UserInfo>query().eq("id_no", updateUserDto.getIdNo()).eq("phone_number",user.getPhone()));
                if(null != oldUserInfo && oldUserInfo.getUserId() == 0L){
                    //如果存在个人报告上传时预先创建的用户信息，且手机号码一致，则将其合并
                    oldUserInfo.setUserId(userId);
                    user.setObjId(oldUserInfo.getId());

                    if(!oldUserInfo.getName().equals(updateUserDto.getName())){
                        updateIMUserProfile(user.getImUserId(),updateUserDto.getName());
                    }
                    oldUserInfo.setName(updateUserDto.getName());
                    oldUserInfo.setSex(updateUserDto.getSex());
                    oldUserInfo.setIdNo(updateUserDto.getIdNo());
                    oldUserInfo.setBornDate(updateUserDto.getBornDate());
                    oldUserInfo.setPhoneNumber(user.getPhone());
                    oldUserInfo.setIdCardPictureBack(updateUserDto.getIdCardPictureBack());
                    oldUserInfo.setIdCardPictureFront(updateUserDto.getIdCardPictureFront());
                    oldUserInfo.setPersonalInfoFilled(YesNoEnum.NO);


                    if(checkInfoFilled(oldUserInfo)){
                        oldUserInfo.setPersonalInfoFilled(YesNoEnum.YES);
                    }
                    userInfoService.saveOrUpdate(oldUserInfo);
                    userDto.setUserInfo(oldUserInfo);

                }else {
                    //全新的用户
                    userInfo = new UserInfo();

                    userInfo.setUserId(userId);
                    //更新IM系统用户信息
                    if(!userInfo.getName().equals(updateUserDto.getName())){
                        updateIMUserProfile(user.getImUserId(),updateUserDto.getName());
                    }
                    userInfo.setName(updateUserDto.getName());
                    userInfo.setSex(updateUserDto.getSex());
                    userInfo.setIdNo(updateUserDto.getIdNo());
                    userInfo.setBornDate(updateUserDto.getBornDate());
                    userInfo.setPhoneNumber(user.getPhone());
                    userInfo.setIdCardPictureBack(updateUserDto.getIdCardPictureBack());
                    userInfo.setIdCardPictureFront(updateUserDto.getIdCardPictureFront());
                    userInfo.setPersonalInfoFilled(YesNoEnum.NO);


                    if(checkInfoFilled(userInfo)){
                        userInfo.setPersonalInfoFilled(YesNoEnum.YES);
                    }
                    userInfoService.saveOrUpdate(userInfo);

                    if(updateUserDto.getDefaultFlag()){
                        PatientInfo patientInfo = patientInfoService.getOne(Wrappers.<PatientInfo>query().eq("user_info_id", userInfo.getId())
                                .eq("relation", PatientRelationEnum.SELF));
                        if(null == patientInfo){
                            //同时创建自身的就诊人
                            patientInfo = new PatientInfo();
                            patientInfo.setUserInfoId(userInfo.getId());
                            patientInfo.setRelation(PatientRelationEnum.SELF);
                            patientInfo.setPatientName(userInfo.getName());
                            patientInfo.setSex(userInfo.getSex());
                            patientInfo.setIdNo(userInfo.getIdNo());
                            patientInfo.setPhoneNumber(userInfo.getPhoneNumber());
                            patientInfo.setBornDate(userInfo.getBornDate());
                            patientInfoService.save(patientInfo);
                        }
                    }


                    user.setObjId(userInfo.getId());
                    userDto.setUserInfo(userInfo);
                }
            }else {
                //已有user_info用户更新个人信息
                userInfo.setUserId(userId);
                //更新IM系统用户信息
                if(!userInfo.getName().equals(updateUserDto.getName())){
                    updateIMUserProfile(user.getImUserId(),updateUserDto.getName());
                }
                userInfo.setName(updateUserDto.getName());
                userInfo.setSex(updateUserDto.getSex());
                userInfo.setIdNo(updateUserDto.getIdNo());
                userInfo.setBornDate(updateUserDto.getBornDate());
                userInfo.setPhoneNumber(user.getPhone());
                userInfo.setIdCardPictureBack(updateUserDto.getIdCardPictureBack());
                userInfo.setIdCardPictureFront(updateUserDto.getIdCardPictureFront());
                userInfo.setPersonalInfoFilled(YesNoEnum.NO);


                if(checkInfoFilled(userInfo)){
                    userInfo.setPersonalInfoFilled(YesNoEnum.YES);
                }
                userInfoService.saveOrUpdate(userInfo);

                if(updateUserDto.getDefaultFlag()){
                    PatientInfo patientInfo = patientInfoService.getOne(Wrappers.<PatientInfo>query().eq("user_info_id", userInfo.getId())
                            .eq("relation", PatientRelationEnum.SELF));
                    if(null == patientInfo){
                        //同时创建自身的就诊人
                        patientInfo = new PatientInfo();
                        patientInfo.setUserInfoId(userInfo.getId());
                        patientInfo.setRelation(PatientRelationEnum.SELF);
                        patientInfo.setPatientName(userInfo.getName());
                        patientInfo.setSex(userInfo.getSex());
                        patientInfo.setIdNo(userInfo.getIdNo());
                        patientInfo.setPhoneNumber(userInfo.getPhoneNumber());
                        patientInfo.setBornDate(userInfo.getBornDate());
                        patientInfoService.save(patientInfo);
                    }
                }
                user.setObjId(userInfo.getId());
                userDto.setUserInfo(userInfo);
            }

        }else if(updateUserDto.getUserType() == UserTypeEnum.DOCTOR){
            DoctorInfo doctorInfo = doctorInfoService.getOne(Wrappers.<DoctorInfo>query().eq("user_id", userId));
            if(null == doctorInfo){
                doctorInfo = new DoctorInfo();
            }
            doctorInfo.setUserId(userId);

            //更新IM系统用户信息
            if(!doctorInfo.getDoctorName().equals(updateUserDto.getName())){
                updateIMUserProfile(user.getImUserId(),updateUserDto.getName());
            }

            doctorInfo.setDoctorName(updateUserDto.getName());
            doctorInfo.setSex(updateUserDto.getSex());
            doctorInfo.setIdNo(updateUserDto.getIdNo());
            doctorInfo.setBornDate(updateUserDto.getBornDate());
            doctorInfo.setPhoneNumber(user.getPhone());
            doctorInfo.setIdCardPictureBack(updateUserDto.getIdCardPictureBack());
            doctorInfo.setIdCardPictureFront(updateUserDto.getIdCardPictureFront());
            doctorInfo.setExpertiseArea(updateUserDto.getExpertiseArea());
            doctorInfo.setIntroduction(updateUserDto.getIntroduction());
            doctorInfoService.saveOrUpdate(doctorInfo);

            user.setObjId(doctorInfo.getId());
            userDto.setDoctorInfo(doctorInfo);
        }
        user.setAvatar(updateUserDto.getHeadUrl());
        this.saveOrUpdate(user);
        userDto.setUser(user);
        return userDto;
    }

    @Override
    public UserDto wxUpdate(UpdateInfoDto updateUserDto) {
        UserDto userDto = new UserDto();

        //根据小程序登录源，切换相应配置
        if(updateUserDto.getUserType() == UserTypeEnum.PATIENT){
            wxMaService.switchover(patientAppId);
        }else if(updateUserDto.getUserType() == UserTypeEnum.DOCTOR){
            wxMaService.switchover(doctorAppId);
        }
        try {
            WxMaJscode2SessionResult session = wxMaService.getUserService().getSessionInfo(updateUserDto.getCode());
            log.debug(session.getSessionKey());
            log.debug(session.getOpenid());
            User user = this.getOne(Wrappers.<User>query().eq("open_id", session.getOpenid()));
            WxMaUserInfo userInfo = wxMaService.getUserService().getUserInfo(session.getSessionKey(), updateUserDto.getEncryptedData(), updateUserDto.getIvStr());
//            user.setUsername(userInfo.getNickName());
            user.setAvatar(userInfo.getAvatarUrl());
            this.saveOrUpdate(user);
            userDto.setUser(user);
            return userDto;
        } catch (WxErrorException e) {
            log.error("微信登录异常");
        } finally {
            WxMaConfigHolder.remove();//清理ThreadLocal
        }
        return userDto;
    }

    @Override
    public String getPhoneNumber(LoginDto loginDto) {
        //根据小程序登录源，切换相应配置
        if(loginDto.getUserType() == UserTypeEnum.PATIENT){
            wxMaService.switchover(patientAppId);
        }else if(loginDto.getUserType() == UserTypeEnum.DOCTOR){
            wxMaService.switchover(doctorAppId);
        }
        try {
            WxMaPhoneNumberInfo phoneNoInfo = wxMaService.getUserService().getPhoneNoInfo(loginDto.getCode());
            return phoneNoInfo.getPurePhoneNumber();
        } catch (WxErrorException e) {
            log.error("微信登录异常");
        } finally {
            WxMaConfigHolder.remove();//清理ThreadLocal
        }
        return null;
    }

    private boolean checkInfoFilled(UserInfo userInfo){
        if (userInfo == null) {
            return false;
        }

        // 获取对象的所有字段
        Field[] fields = userInfo.getClass().getDeclaredFields();

        // 遍历所有字段
        for (Field field : fields) {
            field.setAccessible(true); // 设置字段可访问，即使是私有字段

            try {
                Object fieldValue = field.get(userInfo); // 获取字段的值

                // 如果字段的值为null，则返回false
                if (fieldValue == null) {
                    return false;
                }
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }

        // 如果所有字段都不为null，则返回true
        return true;
    }

    @Override
    public void addUserToIMSystem(String imUserId,String nickName,String headUrl){
        Retrofit retrofit = new Retrofit.Builder()
                .baseUrl(imHost)
                .addConverterFactory(GsonConverterFactory.create())
                .client(new OkHttpClient.Builder().build())
                .build();
        IMApiService service = retrofit.create(IMApiService.class);

        String adminIMSig = getAdminIMSig();

        BindIMUserReq bindIMUserReq = new BindIMUserReq();
        bindIMUserReq.setUserID(imUserId);
        bindIMUserReq.setNick(nickName);
        bindIMUserReq.setFaceUrl(headUrl);
        log.info("正在为用户生成IM系统权限，用户id{}",imUserId);
        Call<JsonObject> call = service.addUserToIMSystem(bindIMUserReq,sdkAppId,"administrator",adminIMSig,"9999","json");
        try {
            Response<JsonObject> execute = call.execute();
            log.info("返回结果{}",execute.body().toString());
            log.info("绑定成功---------");
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void updateIMUserProfile(String imUserId,String nickName){
        Retrofit retrofit = new Retrofit.Builder()
                .baseUrl(imHost)
                .addConverterFactory(GsonConverterFactory.create())
                .client(new OkHttpClient.Builder().build())
                .build();
        IMApiService service = retrofit.create(IMApiService.class);

        String adminIMSig = getAdminIMSig();

        UpdateIMUserReq updateIMUserReq = new UpdateIMUserReq();
        updateIMUserReq.setFrom_Account(imUserId);
        ProfileItem profileItem = new ProfileItem();
        profileItem.setTag("Tag_Profile_IM_Nick");
        profileItem.setValue(nickName);
        ArrayList<ProfileItem> profileItems = new ArrayList<>();
        profileItems.add(profileItem);
        updateIMUserReq.setProfileItem(profileItems);
        log.info("正在为用户生成IM系统权限，用户id{}",imUserId);
        Call<JsonObject> call = service.updateUserProfile(updateIMUserReq,sdkAppId,"administrator",adminIMSig,"9999","json");
        try {
            Response<JsonObject> execute = call.execute();
            log.info("返回结果{}",execute.body().toString());
            log.info("绑定成功---------");
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getIMSig(Long userId){
        //当前用户
        User sysUser = this.getById(userId);
        //先从redis中取缓存没有就生成并存入
        String imSig = redisService.get(Constant.IM_USER_SIG_PREFIX + sysUser.getImUserId());
        if (StringUtils.isEmpty(imSig)){
            TLSSigAPIv2 tlsSigAPIv2 = new TLSSigAPIv2(sdkAppId,key);
            imSig = tlsSigAPIv2.genUserSig(sysUser.getImUserId(), expire);
            redisService.setExpire(Constant.IM_USER_SIG_PREFIX + sysUser.getImUserId(),imSig,expire, TimeUnit.SECONDS);
        }
        return imSig;

    }

    private String getAdminIMSig(){
        //先从redis中取缓存没有就生成并存入
        String imSig = redisService.get(Constant.IM_USER_SIG_PREFIX + "administrator");
        if (StringUtils.isEmpty(imSig)){
            TLSSigAPIv2 tlsSigAPIv2 = new TLSSigAPIv2(sdkAppId,key);
            imSig = tlsSigAPIv2.genUserSig("administrator", expire);
            redisService.setExpire(Constant.IM_USER_SIG_PREFIX + "administrator",imSig,expire, TimeUnit.SECONDS);
        }
        return imSig;

    }

}
